﻿/*
 * Copyright 2019,2020,2021 Sony Corporation
 */

using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Rendering;

using SRD.XR.Core;
using SRD.XR.Utils;

namespace SRD.Sample.Common
{
    public class SRDFaceTrackStateMonitor : MonoBehaviour
    {
        public GameObject FaceTrackErrorObject;

        public int ContinuousFailCountThreshForStartEffect = 50;
        public int ContinuousSuccessCountThreshForBreakEffect = 20;

        public float FaceTrackErrorEffectFadeInSec = 3f;
        public float FaceTrackErrorEffectFadeBackSec = 0.5f;

        private SRDisplayXRManager _srdManager;

        private Material _faceTrackErrorObjMat;
        private Coroutine _fadingCoroutine = null;

        private bool _isSuccessing = true;
        private int _faceTrackContinuousFailCounter = 0;
        private int _faceTrackContinuousSuccessCounter = 0;

        private float _currentErrorObjAlpha;
        private float _defaultErrorObjAlpha;

        private float _errorEffectUpdateDeltaSec = 0.01f;

        void Start()
        {
            _srdManager = SRDisplayXRSceneEnvironment.GetSRDisplayXRManager();
            _srdManager?.OnFaceTrackStateEvent.AddListener(this.GetFaceTrackState);

            FaceTrackErrorObject.SetActive(false);
            _faceTrackErrorObjMat = FaceTrackErrorObject.GetComponent<MeshRenderer>().material;
            _defaultErrorObjAlpha = _currentErrorObjAlpha = _faceTrackErrorObjMat.GetFloat("_Alpha");
        }

        void OnDisable()
        {
            _srdManager?.OnFaceTrackStateEvent.RemoveListener(this.GetFaceTrackState);
        }

        public void GetFaceTrackState(bool isSuccess)
        {
            if (_isSuccessing)
            {
                _faceTrackContinuousFailCounter = isSuccess ? 0 : (_faceTrackContinuousFailCounter + 1);
                if (ContinuousFailCountThreshForStartEffect < _faceTrackContinuousFailCounter)
                {
                    _isSuccessing = false;
                    _faceTrackContinuousSuccessCounter = 0;
                    StartFaceTrackErrorEffect();
                }
            }
            else
            {
                _faceTrackContinuousSuccessCounter = !isSuccess ? 0 : (_faceTrackContinuousSuccessCounter + 1);
                if (ContinuousSuccessCountThreshForBreakEffect < _faceTrackContinuousSuccessCounter)
                {
                    _isSuccessing = true;
                    _faceTrackContinuousFailCounter = 0;
                    BreakFaceTrackErrorEffect();
                }
            }
        }

        private void StartFaceTrackErrorEffect()
        {
            if (_fadingCoroutine != null)
            {
                StopCoroutine(_fadingCoroutine);
            }

            FaceTrackErrorObject.SetActive(true);
            _fadingCoroutine = StartCoroutine(EffectCoroutine(_defaultErrorObjAlpha, _defaultErrorObjAlpha,
                                                              FaceTrackErrorEffectFadeInSec, () => {}));
        }

        private void BreakFaceTrackErrorEffect()
        {
            if (_fadingCoroutine != null)
            {
                StopCoroutine(_fadingCoroutine);
            }
            _fadingCoroutine = StartCoroutine(EffectCoroutine(_currentErrorObjAlpha, 0.0f,
                                                              FaceTrackErrorEffectFadeBackSec, () =>
                                                              {
                                                                  FaceTrackErrorObject.SetActive(false);
                                                              }));
        }

        private IEnumerator EffectCoroutine(float fromAlpha, float toAlpha, float durationSec, Action onFinished)
        {
            SetErrorObjAlpha(fromAlpha);

            var yielder = new WaitForSeconds(_errorEffectUpdateDeltaSec);
            var elaspedTime = 0f;
            var startTime = Time.time;
            yield return null;
            while (elaspedTime < durationSec)
            {
                yield return yielder;
                elaspedTime = Time.time - startTime;
                SetErrorObjAlpha(fromAlpha + (toAlpha - fromAlpha) * (elaspedTime / durationSec));
            }
            SetErrorObjAlpha(toAlpha);
            if (onFinished != null)
            {
                onFinished();
            }
        }

        private void SetErrorObjAlpha(float alpha)
        {
            _currentErrorObjAlpha = alpha;
            _faceTrackErrorObjMat.SetFloat("_Alpha", _currentErrorObjAlpha);
        }
    }
}
