﻿/*
 * Copyright 2019,2020,2021 Sony Corporation
 */

using UnityEngine;
using SRD.XR.Utils;

namespace SRD.Sample.Raycast
{
    public class SRD3DRaycastSample : MonoBehaviour
    {
        public GameObject PointerObject;
        [ColorUsage(false, true)]
        public Color DraggingColor = Color.cyan;

        private SRDisplayXRCamera _srdCameras;

        private ColorToggler _colorToggler;

        private bool _isDragging = false;
        private GameObject _targetObjectCache;
        private Vector3 _mousePosInWorldCache;

        private readonly int LeftMouseButton = 0;

        void Start()
        {
            _srdCameras = new SRDisplayXRCamera();
            _colorToggler = new ColorToggler(DraggingColor);

            if(PointerObject != null)
            {
                Cursor.visible = false;
            }
        }

        void OnDisable()
        {
            Cursor.visible = true;
        }

        void Update()
        {
            var mousePosInScreen = Input.mousePosition;
            var mousePosInWorld = _srdCameras.ScreenToWorldPoint(mousePosInScreen);

            var ray = _srdCameras.ScreenPointToRay(mousePosInScreen);
            RaycastHit hit;
            if(!Physics.Raycast(ray, out hit))
            {
                if(_isDragging)
                {
                    _colorToggler.TurnOff(_targetObjectCache.GetComponent<MeshRenderer>());
                }
                if(PointerObject != null)
                {
                    PointerObject.transform.position = mousePosInWorld;
                }
                return;
            }

            if(PointerObject != null)
            {
                PointerObject.transform.position = hit.point;
            }
            MoveTarget(hit.collider.gameObject, mousePosInWorld);
        }

        private void MoveTarget(GameObject target, Vector3 mousePosInWorld)
        {
            var mouseDownThisFrame = Input.GetMouseButtonDown(LeftMouseButton);
            var mouseUpThisFrame = Input.GetMouseButtonUp(LeftMouseButton);
            var mousePressed = Input.GetMouseButton(LeftMouseButton);

            _targetObjectCache = target;

            if(mouseDownThisFrame)
            {
                _isDragging = true;
                _colorToggler.TurnOn(_targetObjectCache.GetComponent<MeshRenderer>());
                _mousePosInWorldCache = mousePosInWorld;
            }
            else if(mouseUpThisFrame)
            {
                _isDragging = false;
                _colorToggler.TurnOff(_targetObjectCache.GetComponent<MeshRenderer>());
            }
            else if(mousePressed)
            {
                var diff = mousePosInWorld - _mousePosInWorldCache;

                var watcherTransform = Camera.main.transform;
                var diffInWatcherCoord = watcherTransform.InverseTransformVector(diff);
                diffInWatcherCoord.z = 0;

                _targetObjectCache.transform.position += watcherTransform.TransformVector(diffInWatcherCoord);

                _mousePosInWorldCache = mousePosInWorld;
            }
        }

        class ColorToggler
        {
            private Color _defaultColorCache;
            private Color _defaultEmissiveColorCache;
            private Color _turnOnColor;
            private bool _isOn = false;
            public ColorToggler(Color turnOnColor)
            {
                _turnOnColor = turnOnColor;
            }

            public void TurnOn(MeshRenderer mr)
            {
                if(_isOn)
                {
                    return;
                }
                mr.material.shader = Shader.Find("HDRP/Lit");
                _defaultColorCache = mr.material.GetColor("_BaseColor");
                _defaultEmissiveColorCache = mr.material.GetColor("_EmissiveColor");
                mr.material.SetColor("_BaseColor", _turnOnColor);
                mr.material.SetColor("_EmissiveColor", _turnOnColor);
                _isOn = true;
            }

            public void TurnOff(MeshRenderer mr)
            {
                if(!_isOn)
                {
                    return;
                }
                mr.material.shader = Shader.Find("HDRP/Lit");
                mr.material.SetColor("_BaseColor", _defaultColorCache);
                mr.material.SetColor("_EmissiveColor", _defaultEmissiveColorCache);
                _isOn = false;
            }
        }
    }

}
