﻿/*
 * Copyright 2019,2020,2021,2022 Sony Corporation
 */

using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.Rendering;
using UnityEngine.Events;
using UnityEngine.XR;
using UnityEngine.XR.Management;

namespace SRD.XR.Core
{
    [DisallowMultipleComponent]
    public class SRDisplayXRManager : MonoBehaviour
    {
        /// <summary>
        /// A flag for SR Rendering
        /// </summary>
        /// <remarks>
        /// If this is disabled, SR Rendering is turned off.
        /// </remarks>
        [Tooltip("If this is disabled, SR Rendering is turned off.")]
        public bool IsSRRenderingActive = true;

        /// <summary>
        /// A flag for Crosstalk Correction
        /// </summary>
        /// <remarks>
        /// If this is disabled, the crosstalk correction is turned off.
        /// </remarks>
        [Tooltip("If this is disabled, the crosstalk correction is turned off.")]
        public bool IsCrosstalkCorrectionActive = true;

        /// <summary>
        /// Crosstalk Correction Type
        /// </summary>
        /// <remarks>
        /// This is valid only if the crosstalk correction is active.
        /// </remarks>
        [Tooltip("This is valid only if the crosstalk correction is active.")]
        public SrdXrCrosstalkCorrectionType CrosstalkCorrectionType;

        private const float _minScaleInInspector = 0.1f;
        private const float _maxScaleInInspector = 1000.0f; // float.MaxValue;
        [SerializeField]
        [Range(_minScaleInInspector, _maxScaleInInspector), Tooltip("The scale of SRDisplay View Space")]
        private float _SRDViewSpaceScale = 1.0f;

        /// <summary>
        /// The scale of SRDisplay View Space
        /// </summary>
        public float SRDViewSpaceScale
        {
            get
            {
                return _SRDViewSpaceScale;
            }
            set
            {
                if (value <= 0)
                {
                    Debug.LogWarning(String.Format("Wrong SRDViewSpaceScale: {0} \n SRDViewSpaceScale must be 0+. Now SRDViewSpaceScale is {1}.", value, _SRDViewSpaceScale));
                }
                else
                {
                    _SRDViewSpaceScale = value;
                }
            }
        }

        private Transform _srdMainCamTransform;

        #region Events
        /// <summary>
        /// A UnityEvent callback containing SRDisplayViewSpaceScale.
        /// </summary>
        [System.Serializable]
        public class SRDViewSpaceScaleChangedEvent : UnityEvent<float> { };
        /// <summary>
        /// An API of <see cref="SRDisplayXRManager.SRDViewSpaceScaleChangedEvent"/>. Callbacks that are registered to this are called when SRDViewSpaceScale is changed.
        /// </summary>
        public SRDViewSpaceScaleChangedEvent OnSRDViewSpaceScaleChangedEvent;

        /// <summary>
        /// A UnityEvent callback containing a flag that describe FaceTrack is success or not in this frame.
        /// </summary>
        [System.Serializable]
        public class SRDFaceTrackStateEvent : UnityEvent<bool> { };
        /// <summary>
        /// An API of <see cref="SRDisplayXRManager.SRDFaceTrackStateEvent"/>. Callbacks that are registered to this are called in every frame.
        /// </summary>
        public SRDFaceTrackStateEvent OnFaceTrackStateEvent;

        #endregion

        #region APIs
        public void ShowCameraWindow(bool isOn)
        {
            SRDisplayXrCorePlugin.ShowCameraWindow(isOn);
        }

        #endregion

        #region MainFlow
        void Awake()
        {
            Application.targetFrameRate = 1000;
            GetSRDisplayXRCameraOrAddIfNeeded(out _srdMainCamTransform);

            SRDisplayXrCorePlugin.Setup();
            SRDisplayXrCorePlugin.SetFaceTrackStateEvent((tracked) => 
            {
                if (OnFaceTrackStateEvent != null)
                {
                    OnFaceTrackStateEvent.Invoke(tracked);
                }
            });

            UpdatePlugin();
        }

        IEnumerator Start()
        {
            while(!SplashScreen.isFinished)
            {
                yield return null;
            }

            var loader = XRGeneralSettings.Instance.Manager.ActiveLoaderAs<SRDisplayXRLoader>();
            loader?.Start();
        }

        void Update()
        {
            UpdateScaleIfNeeded();
            UpdateSRDisplayXRCamera();
            UpdatePlugin();
        }

        void OnValidate()
        {
            UpdateScaleIfNeeded();
        }

        void OnDestroy()
        {
            SRDisplayXrCorePlugin.Cleanup();
        }

        #endregion


        #region Utils
        private void UpdatePlugin()
        {
            SRDisplayXrCorePlugin.SetActiveStateCrosstalkCorrection(IsCrosstalkCorrectionActive, CrosstalkCorrectionType);
            SRDisplayXrCorePlugin.SetEnabledPassThrough(!IsSRRenderingActive);
            var settings = new SrdXrColorManagementSettings();
            var unityGamma = 2.2f;
            settings.gamma = unityGamma;
            settings.is_input_texture_gamma_corrected = QualitySettings.desiredColorSpace == ColorSpace.Gamma;
            SRDisplayXrCorePlugin.SetColorManagementSettings(settings);
        }

        private void GetSRDisplayXRCameraOrAddIfNeeded(out Transform srdCamTransform)
        {
            var mainCam = Camera.main;
            if (mainCam == null)
            {
                var go = new GameObject(SRDisplayXRUtils.Constrains.MainCamName);
                go.transform.parent = this.transform;
                mainCam = go.AddComponent<Camera>();
                mainCam.tag = "Main Camera";
            }
            srdCamTransform = mainCam.gameObject.transform;
        }

        private void UpdateSRDisplayXRCamera()
        {
            var nodeStates = new List<XRNodeState>();
            InputTracking.GetNodeStates(nodeStates);
            foreach (var nodeState in nodeStates)
            {
                if (nodeState.nodeType != XRNode.CenterEye)
                {
                    continue;
                }

                var currentPose = new Pose(_srdMainCamTransform.localPosition, _srdMainCamTransform.localRotation);
                if (nodeState.TryGetPosition(out currentPose.position))
                {
                    _srdMainCamTransform.localPosition = currentPose.position;
                }
                if (nodeState.TryGetRotation(out currentPose.rotation))
                {
                    _srdMainCamTransform.localRotation = currentPose.rotation;
                }
                return;
            }
        }

        private void UpdateScaleIfNeeded()
        {
            if (this.transform.localScale != Vector3.one * SRDViewSpaceScale)
            {
                this.transform.localScale = Vector3.one * SRDViewSpaceScale;
                if (OnSRDViewSpaceScaleChangedEvent != null)
                {
                    OnSRDViewSpaceScaleChangedEvent.Invoke(SRDViewSpaceScale);
                }
            }
        }

        #endregion

        #region GIZMO

        private void OnDrawGizmos()
        {
            if (!this.enabled)
            {
                return;
            }

            var bodyBounds = new SRDisplayXRUtils.BodyBounds();

            // Draw SRDisplay View Space
            Gizmos.matrix = transform.localToWorldMatrix;
            Gizmos.color = Color.blue;
            Gizmos.DrawWireCube(bodyBounds.Center, bodyBounds.BoxSize);
            Gizmos.color = Color.cyan;
            for (var i = 0; i < 4; i++)
            {
                var from = i % 4;
                var to = (i + 1) % 4;
                Gizmos.DrawLine(bodyBounds.EdgePositions[from], bodyBounds.EdgePositions[to]);
            }
        }
        #endregion
    }

}
