﻿/*
 * Copyright 2021 Sony Corporation
 */

using System;
using System.Runtime.InteropServices;

using SRD.XR.Utils;

namespace SRD.XR.Core
{
    internal static class SRDisplayXrCorePlugin
    {
        public const SrdXrCrosstalkCorrectionType DefaultCrosstalkCorrectionType = SrdXrCrosstalkCorrectionType.Medium;

        private static Action<bool> OnFaceTrackStateEvent;

        public static void Setup()
        {
            SRDisplayXrRuntimeAPI.SetFaceTrackStateEvent(FaceTrackStateEventCallback);
        }

        public static void Cleanup()
        {
            OnFaceTrackStateEvent = null;
            SRDisplayXrRuntimeAPI.SetFaceTrackStateEvent(null);
        }

        public static int ShowMessageBox(string title, string message, Action<string> debugLogFunc = null)
        {
            if (debugLogFunc != null)
            {
                debugLogFunc(message);
            }
            return SRDisplayXrRuntimeAPI.ShowMessageBox(SRDisplayXRApplicationWindow.GetSelfWindowHandle(), title, message);
        }

        public static void SetActiveStateCrosstalkCorrection(bool state, SrdXrCrosstalkCorrectionType type = DefaultCrosstalkCorrectionType)
        {
            SRDisplayXrRuntimeAPI.SetCorrectionMode(state, type);
        }

        public static void SetEnabledPassThrough(bool enabled)
        {
            SRDisplayXrRuntimeAPI.SetEnabledPassThrough(enabled);
        }

        public static void SetColorManagementSettings(SrdXrColorManagementSettings settings)
        {
            SRDisplayXrRuntimeAPI.SetColorManagementSettings(ref settings);
        }

        [AOT.MonoPInvokeCallback(typeof(SRDisplayXrRuntimeAPI.FaceTrackStateEventDelegate))]
        private static void FaceTrackStateEventCallback(bool tracked)
        {
            if (OnFaceTrackStateEvent != null)
            {
                OnFaceTrackStateEvent.Invoke(tracked);
            }
        }

        public static void SetFaceTrackStateEvent(Action<bool> trackedCallback)
        {
            OnFaceTrackStateEvent += trackedCallback;
        }

        public static bool GetSRDScreenRect(out SRDisplayXRSettings.ScreenRect screenRect)
        {
            SrdXrDeviceInfo[] devices = { new SrdXrDeviceInfo(), };
            var result = EnumerateDevices(devices, 1);

            if (result != SrdXrResult.SUCCESS)
            {
                screenRect = new SRDisplayXRSettings.ScreenRect();
                return false;
            }
            var target = devices[0].target_monitor_rectangle;
            var width = target.right - target.left;
            var height = target.bottom - target.top;
            screenRect = new SRDisplayXRSettings.ScreenRect(target.left, target.top, width, height);
            return true;
        }

        public static SrdXrResult EnumerateDevices([In, Out] SrdXrDeviceInfo[] devices, UInt32 loadCount)
        {
            var result = SRDisplayXrRuntimeAPI.EnumerateDevices(devices, loadCount);
            if (result != SrdXrResult.SUCCESS)
            {
                for (int i = 0; i < loadCount; i++)
                {
                    devices[i].device_index = 0;
                    devices[i].device_serial_number = "";
                    devices[i].primary_monitor_rectangle = new SrdXrRect(0, 0, 0, 0);
                    devices[i].target_monitor_rectangle = new SrdXrRect(0, 0, 0, 0);
                }
            }
            return result;
        }

        public static void ShowCameraWindow(bool show)
        {
            SRDisplayXrRuntimeAPI.ShowCameraWindow(show);
        }

        private struct SRDisplayXrRuntimeAPI
        {
            const string dll_path = SRDisplayXRUtils.Constants.NativePluginDLLName;

            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_SetCorrectionMode")]
            public static extern void SetCorrectionMode(bool enabled, SrdXrCrosstalkCorrectionType type);

            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_SetEnabledPassThrough")]
            public static extern void SetEnabledPassThrough(bool enabled);

            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_SetColorManagementSettings")]
            public static extern void SetColorManagementSettings([In] ref SrdXrColorManagementSettings settings);

            [UnmanagedFunctionPointer(CallingConvention.Cdecl)]
            public delegate void FaceTrackStateEventDelegate([MarshalAs(UnmanagedType.U1)] bool tracked);
            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_SetFaceTrackStateEvent")]
            public static extern void SetFaceTrackStateEvent(FaceTrackStateEventDelegate face_track_state_event_delegate);

            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_EnumerateDevices")]
            public static extern SrdXrResult EnumerateDevices([In, Out] SrdXrDeviceInfo[] devices, UInt32 load_count);

            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_ShowCameraWindow")]
            public static extern void ShowCameraWindow(bool show);

            [DllImport(dll_path, EntryPoint = "UnityXrPlugin_ShowMessageBox", CharSet = CharSet.Unicode)]
            public static extern int ShowMessageBox(IntPtr hWnd, string title, string msg);
        }
    }

    public enum SrdXrCrosstalkCorrectionType
    {
        Medium = 0,
        All = 1,
        HighPrecise = 2,
    }

    public enum SrdXrResult
    {
        SUCCESS = 0,
        TIMEOUT_EXPIRED = 1,
        SESSION_LOSS_PENDING = 3,
        EVENT_UNAVAILABLE = 4,
        SPACE_BOUNDS_UNAVAILABLE = 7,
        SESSION_NOT_FOCUSED = 8,
        FRAME_DISCARDED = 9,
        ERROR_VALIDATION_FAILURE = -1,
        ERROR_RUNTIME_FAILURE = -2,
        ERROR_OUT_OF_MEMORY = -3,
        ERROR_API_VERSION_UNSUPPORTED = -4,
        ERROR_INITIALIZATION_FAILED = -6,
        ERROR_FUNCTION_UNSUPPORTED = -7,
        ERROR_FEATURE_UNSUPPORTED = -8,
        ERROR_EXTENSION_NOT_PRESENT = -9,
        ERROR_LIMIT_REACHED = -10,
        ERROR_SIZE_INSUFFICIENT = -11,
        ERROR_HANDLE_INVALID = -12,
        ERROR_INSTANCE_LOST = -13,
        ERROR_SESSION_RUNNING = -14,
        ERROR_SESSION_NOT_RUNNING = -16,
        ERROR_SESSION_LOST = -17,
        ERROR_SYSTEM_INVALID = -18,
        ERROR_PATH_INVALID = -19,
        ERROR_PATH_COUNT_EXCEEDED = -20,
        ERROR_PATH_FORMAT_INVALID = -21,
        ERROR_PATH_UNSUPPORTED = -22,
        ERROR_LAYER_INVALID = -23,
        ERROR_LAYER_LIMIT_EXCEEDED = -24,
        ERROR_SWAPCHAIN_RECT_INVALID = -25,
        ERROR_SWAPCHAIN_FORMAT_UNSUPPORTED = -26,
        ERROR_ACTION_TYPE_MISMATCH = -27,
        ERROR_SESSION_NOT_READY = -28,
        ERROR_SESSION_NOT_STOPPING = -29,
        ERROR_TIME_INVALID = -30,
        ERROR_REFERENCE_SPACE_UNSUPPORTED = -31,
        ERROR_FILE_ACCESS_ERROR = -32,
        ERROR_FILE_CONTENTS_INVALID = -33,
        ERROR_FORM_FACTOR_UNSUPPORTED = -34,
        ERROR_FORM_FACTOR_UNAVAILABLE = -35,
        ERROR_API_LAYER_NOT_PRESENT = -36,
        ERROR_CALL_ORDER_INVALID = -37,
        ERROR_GRAPHICS_DEVICE_INVALID = -38,
        ERROR_POSE_INVALID = -39,
        ERROR_INDEX_OUT_OF_RANGE = -40,
        ERROR_VIEW_CONFIGURATION_TYPE_UNSUPPORTED = -41,
        ERROR_ENVIRONMENT_BLEND_MODE_UNSUPPORTED = -42,
        ERROR_NAME_DUPLICATED = -44,
        ERROR_NAME_INVALID = -45,
        ERROR_ACTIONSET_NOT_ATTACHED = -46,
        ERROR_ACTIONSETS_ALREADY_ATTACHED = -47,
        ERROR_LOCALIZED_NAME_DUPLICATED = -48,
        ERROR_LOCALIZED_NAME_INVALID = -49,

        ERROR_RUNTIME_NOT_FOUND = -1001,
        ERROR_DEVICE_NOT_FOUND = -1011,
        ERROR_DISPLAY_NOT_FOUND = -1012,
        ERROR_USB_NOT_CONNECTED = -1013,
        ERROR_USB_3_NOT_CONNECTED = -1014,

        RESULT_MAX_ENUM = 0x7FFFFFFF
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct SrdXrRect
    {
        public int left;
        public int top;
        public int right;
        public int bottom;

        public SrdXrRect(int in_left, int in_top, int in_right, int in_bottom)
        {
            left = in_left;
            top = in_top;
            right = in_right;
            bottom = in_bottom;
        }
    };

    [StructLayout(LayoutKind.Sequential, CharSet = CharSet.Unicode)]
    public struct SrdXrDeviceInfo
    {
        public UInt32 device_index;
        [MarshalAs(UnmanagedType.ByValTStr, SizeConst = 256)] public string device_serial_number;
        public SrdXrRect target_monitor_rectangle;
        public SrdXrRect primary_monitor_rectangle;

        public SrdXrDeviceInfo(UInt32 in_device_index, string in_device_serial_number
                               , SrdXrRect in_target_monitor_rectangle, SrdXrRect in_primary_monitor_rectangle)
        {
            device_index = in_device_index;
            device_serial_number = in_device_serial_number;
            target_monitor_rectangle = in_target_monitor_rectangle;
            primary_monitor_rectangle = in_primary_monitor_rectangle;
        }
    };

    [StructLayout(LayoutKind.Sequential)]
    public struct SrdXrColorManagementSettings
    {
        public bool is_input_texture_gamma_corrected;
        public float gamma;

        public SrdXrColorManagementSettings(bool in_is_input_texture_gamma_corrected, float in_gamma)
        {
            is_input_texture_gamma_corrected = in_is_input_texture_gamma_corrected;
            gamma = in_gamma;
        }
    };
}
