﻿/*
 * Copyright 2021 Sony Corporation
 */

using System.Collections.Generic;
using UnityEngine;

using SRD.XR.Core;

namespace SRD.XR.Utils
{
    public class SRDisplayXRCamera
    {
        private DisplayEdges _displayEdges;
        private Camera _mainCamera;
        public SRDisplayXRCamera()
        {
            _displayEdges = new DisplayEdges();
            _mainCamera = Camera.main;
        }

        public Ray ScreenPointToRay(Vector2 positionInScreen)
        {
            var posInWorld = ScreenToWorldPoint(positionInScreen);
            var cameraPosition = _mainCamera.transform.position;
            return new Ray(cameraPosition, (posInWorld - cameraPosition).normalized);
        }

        /// <summary>
        /// Transform a point from screen space into world space.
        /// A return position must be on the Gizmo cyan wireframe plane from <see cref="SRDManager"/>
        /// Similar to <a href="https://docs.unity3d.com/ScriptReference/Camera.ScreenToWorldPoint.html">Camera.ScreenToWorldPoint</a>
        /// </summary>
        /// <param name="positionInScreen">Position in screen space to get from Unity API. For example, <a href="https://docs.unity3d.com/ScriptReference/Input-mousePosition.html">Input.mousePosition</a> </param>
        /// <returns>The world space point converted from the screen space point</returns>
        public Vector3 ScreenToWorldPoint(Vector2 positionInScreen)
        {
            var screenRate = new Vector2(positionInScreen.x / Screen.width, positionInScreen.y / Screen.height);
            var edges = _displayEdges.Positions;
            var xDirection = edges[2] - edges[1];
            var yDirection = edges[0] - edges[1];
            return edges[1] + xDirection * screenRate.x + yDirection * screenRate.y;
        }

        /// <summary>
        /// Transform a point from world space into actual Spatial Reality Display screen space.
        /// </summary>
        /// <param name="positionInWorld">Position in world space</param>
        /// <returns>A position in Spatial Reality Display screen space</returns>
        public Vector2 WorldToSRDScreenPoint(Vector3 positionInWorld)
        {
            return ScreenToSRDScreen(_mainCamera.WorldToScreenPoint(positionInWorld));
        }

        /// <summary>
        /// Transform a point from screen space that Unity provides into actual Spatial Reality Display screen space.
        /// The inverse transform of <see cref="SRDCameras.SRDScreenToScreen">
        /// </summary>
        /// <param name="positionInScreen">Position in screen space to get from Unity API. For example, <a href="https://docs.unity3d.com/ScriptReference/Input-mousePosition.html">Input.mousePosition</a></param>
        /// <returns>A position in Spatial Reality Display screen space</returns>
        public Vector2 ScreenToSRDScreen(Vector2 positionInScreen)
        {
            var h = CalcHomographyMatrix(_displayEdges.LeftUp.position,
                                         _displayEdges.LeftBottom.position,
                                         _displayEdges.RightBottom.position,
                                         _displayEdges.RightUp.position,
                                         _mainCamera);
            var x = positionInScreen.x / Screen.width;
            var y = positionInScreen.y / Screen.height;
            var w = h[6] * x + h[7] * y + h[8];
            return new Vector2(Screen.width * (h[0] * x + h[1] * y + h[2]) / w,
                               Screen.height * (h[3] * x + h[4] * y + h[5]) / w);

        }

        /// <summary>
        /// Transform a point from actual Spatial Reality Display screen space into rendering camera based screen space.
        /// The inverse transform of <see cref="SRDCameras.ScreenToSRDScreen">
        /// </summary>
        /// <param name="positionInSRDScreen">Position in Spatial Reality Display screen space </param>
        /// <returns>Position in rendering camera based screen space</returns>
        public Vector2 SRDScreenToScreen(Vector2 positionInSRDScreen)
        {
            var h = CalcHomographyMatrix(_displayEdges.LeftUp.position,
                                         _displayEdges.LeftBottom.position,
                                         _displayEdges.RightBottom.position,
                                         _displayEdges.RightUp.position,
                                         _mainCamera);
            h = CalcInverseMatrix3x3(h);
            var x = positionInSRDScreen.x / Screen.width;
            var y = positionInSRDScreen.y / Screen.height;
            var w = h[6] * x + h[7] * y + h[8];
            return new Vector2(Screen.width * (h[0] * x + h[1] * y + h[2]) / w,
                               Screen.height * (h[3] * x + h[4] * y + h[5]) / w);

        }

        private float[] CalcHomographyMatrix(Vector3 leftUp, Vector3 leftBottom, Vector3 rightBottom, Vector3 rightUp, UnityEngine.Camera camera)
        {
            Vector2 p00 = camera.WorldToViewportPoint(leftBottom);
            Vector2 p01 = camera.WorldToViewportPoint(leftUp);
            Vector2 p10 = camera.WorldToViewportPoint(rightBottom);
            Vector2 p11 = camera.WorldToViewportPoint(rightUp);

            var x00 = p00.x;
            var y00 = p00.y;
            var x01 = p01.x;
            var y01 = p01.y;
            var x10 = p10.x;
            var y10 = p10.y;
            var x11 = p11.x;
            var y11 = p11.y;

            var a = x10 - x11;
            var b = x01 - x11;
            var c = x00 - x01 - x10 + x11;
            var d = y10 - y11;
            var e = y01 - y11;
            var f = y00 - y01 - y10 + y11;

            var h13 = x00;
            var h23 = y00;
            var h32 = (c * d - a * f) / (b * d - a * e);
            var h31 = (c * e - b * f) / (a * e - b * d);
            var h11 = x10 - x00 + h31 * x10;
            var h12 = x01 - x00 + h32 * x01;
            var h21 = y10 - y00 + h31 * y10;
            var h22 = y01 - y00 + h32 * y01;

            return new float[] { h11, h12, h13, h21, h22, h23, h31, h32, 1f };
        }

        private float[] CalcInverseMatrix3x3(float[] mat)
        {
            var i11 = mat[0];
            var i12 = mat[1];
            var i13 = mat[2];
            var i21 = mat[3];
            var i22 = mat[4];
            var i23 = mat[5];
            var i31 = mat[6];
            var i32 = mat[7];
            var i33 = mat[8];
            var a = 1f / (
                        +(i11 * i22 * i33)
                        + (i12 * i23 * i31)
                        + (i13 * i21 * i32)
                        - (i13 * i22 * i31)
                        - (i12 * i21 * i33)
                        - (i11 * i23 * i32)
                    );

            var o11 = (i22 * i33 - i23 * i32) / a;
            var o12 = (-i12 * i33 + i13 * i32) / a;
            var o13 = (i12 * i23 - i13 * i22) / a;
            var o21 = (-i21 * i33 + i23 * i31) / a;
            var o22 = (i11 * i33 - i13 * i31) / a;
            var o23 = (-i11 * i23 + i13 * i21) / a;
            var o31 = (i21 * i32 - i22 * i31) / a;
            var o32 = (-i11 * i32 + i12 * i31) / a;
            var o33 = (i11 * i22 - i12 * i21) / a;

            return new float[] { o11, o12, o13, o21, o22, o23, o31, o32, o33 };
        }

        private class DisplayEdges
        {
            public DisplayEdges(Transform leftUp, Transform leftBottom, Transform rightBottom, Transform rightUp)
            {
                this.leftUp = leftUp;
                this.leftBottom = leftBottom;
                this.rightUp = rightUp;
                this.rightBottom = rightBottom;
            }

            public DisplayEdges()
            {
                var bodyBounds = new SRDisplayXRUtils.BodyBounds();
                var srdXrmanager = SRDisplayXRSceneEnvironment.GetSRDisplayXRManager();
                var dispEdges = new List<GameObject>();
                foreach (var edge in bodyBounds.EdgePositions)
                {
                    var go = new GameObject();
                    go.transform.SetParent(srdXrmanager.transform);
                    go.transform.localPosition = edge;
                    go.transform.localRotation = Quaternion.identity;
                    go.hideFlags = HideFlags.HideAndDontSave;
                    dispEdges.Add(go);
                }
                this.leftUp = dispEdges[0].transform;
                this.leftBottom = dispEdges[1].transform;
                this.rightBottom = dispEdges[2].transform;
                this.rightUp = dispEdges[3].transform;
            }

            private Transform leftUp;
            public Transform LeftUp { get { return leftUp; } }

            private Transform leftBottom;
            public Transform LeftBottom { get { return leftBottom; } }

            private Transform rightUp;
            public Transform RightUp { get { return rightUp; } }

            private Transform rightBottom;
            public Transform RightBottom { get { return rightBottom; } }

            public Vector3[] Positions
            {
                get
                {
                    return new Vector3[]
                    {
                    this.leftUp.position, this.leftBottom.position,
                    this.rightBottom.position, this.rightUp.position
                    };
                }
            }
        }

    }
}
