﻿/*
 * Copyright 2019,2020,2022 Sony Corporation
 */

using UnityEngine;
using UnityEditor;
using UnityEditor.Build;
using UnityEditor.Build.Reporting;

namespace SRD.XR.Editor
{
    internal class SRDisplayXRSettingsChecker : IPreprocessBuildWithReport, IActiveBuildTargetChanged
    {
        public int callbackOrder { get { return 0; } }

        [UnityEditor.InitializeOnLoadMethod]
        public static void CheckBuildTargetOnLoad()
        {
            EditorApplication.playModeStateChanged += OnPlayModeStateChanged;

            ForceSwitchBuildTargetIfNeeded(EditorUserBuildSettings.activeBuildTarget);
        }

        private static void OnPlayModeStateChanged(PlayModeStateChange state)
        {
            if(state == PlayModeStateChange.EnteredPlayMode)
            {
                if(SystemInfo.graphicsDeviceType != UnityEngine.Rendering.GraphicsDeviceType.Direct3D11)
                {
                    Debug.LogAssertionFormat("Current Graphics API({0}) is NOT supported. Switch to Direct3D11.", SystemInfo.graphicsDeviceType);
                    UnityEditor.EditorApplication.isPlaying = false;
                    return;
                }

                Application.runInBackground = true;
            }
        }

        public void OnPreprocessBuild(BuildReport report)
        {
            if(report.summary.platform != BuildTarget.StandaloneWindows64)
            {
                throw new BuildFailedException(string.Format("Current BuildPlatform({0}) is NOT supported. Switch to Windows x64", report.summary.platform));
            }

            var graphicsAPIs = PlayerSettings.GetGraphicsAPIs(report.summary.platform);
            if (graphicsAPIs[0] != UnityEngine.Rendering.GraphicsDeviceType.Direct3D11)
            {
                throw new BuildFailedException(string.Format("Current Graphics API({0}) is NOT supported. Switch to Direct3D11.", graphicsAPIs[0]));
            }

            Application.runInBackground = true;
        }

        public void OnActiveBuildTargetChanged(BuildTarget previousTarget, BuildTarget newTarget)
        {
            ForceSwitchBuildTargetIfNeeded(newTarget);
        }

        public static void ForceSwitchBuildTargetIfNeeded(BuildTarget currentTarget)
        {
            if(currentTarget != BuildTarget.StandaloneWindows64)
            {
                var message = "Spatial Reality Display supports Windows x64 only. Force to switch build target to Windows x64.";
                if(EditorUtility.DisplayDialog("Confirm", message, "OK"))
                {
                    EditorApplication.update += WaitForSwitchBuildTargetComplete;
                    Debug.Log(message);
                }
                else
                {
                    Debug.LogError(string.Format("Current BuildPlatform({0}) is NOT supported. Switch to Windows x64", currentTarget));
                }
            }

        }

        static void WaitForSwitchBuildTargetComplete()
        {

            var result = EditorUserBuildSettings.SwitchActiveBuildTargetAsync(BuildTargetGroup.Standalone, BuildTarget.StandaloneWindows64);
            if(result)
            {
                EditorApplication.update -= WaitForSwitchBuildTargetComplete;
            }
        }

    }
}
