/*
 * Copyright 2019,2020,2021 Sony Corporation
 */

#pragma once

#include <CoreMinimal.h>

#include "SRDisplayManager.generated.h"

class USRDisplayDrawBoxComponent;

UENUM()
enum class ECrosstalkCorrectionType
{
	/*
	Corrects crosstalk and make it less noticeable at medium gradation.
	GPU load will be a little higher than when crosstalk correction is not used.
	*/
	GRADATION_CORRECTION_MEDIUM UMETA(DisplayName = "Medium gradation correction"),
	/*
	Corrects crosstalk and make it less noticeable at all gradation.
	GPU load will be higher than that of "Medium gradation correction".
	*/
	GRADATION_CORRECTION_ALL UMETA(DisplayName = "All gradation correction​"),
	/*
	Corrects crosstalk at all gradation.
	Crosstalk will be less noticeable than "All gradation correction".
	GPU load will be higher than when "All gradation correction".
	*/
	GRADATION_CORRECTION_HIGH_PRECISE UMETA(DisplayName = "All gradation correction (High Precise)​")
};

UCLASS()
class ASRDisplayManager : public AActor
{
	GENERATED_BODY()

public:

	ASRDisplayManager();

	// Tracking
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Tracking",
		meta = (ToolTip = "If checked, a window opens and shows images captured by SRD camera."))
	bool ShowCameraWindow = false;

	// Rendering
	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Rendering|Crosstalk Correction",
		meta = (ToolTip = "If checked, crosstalk will be reduced."))
	bool CrosstalkCorrection = true;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Rendering|Crosstalk Correction",
		meta = (ToolTip = "Crosstalk Correction level can be selected.", EditCondition = "CrosstalkCorrection"))
	ECrosstalkCorrectionType CorrectionType = ECrosstalkCorrectionType::GRADATION_CORRECTION_MEDIUM;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Rendering",
		meta = (ToolTip = "If checked, contents will be clipped at front of SR Display."))
	bool SpatialClipping = false;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Rendering",
		meta = (DisplayName = "Far Clip (cm)", ClampMin = "0.0", UIMin = "0.0", ToolTip = "Far clipt distance (cm)."))
	float FarClip = 100000000.f;

	// Appearance
	UPROPERTY(EditAnywhere, BlueprintReadWrite, SimpleDisplay, Category = "SRDisplay|Appearance")
	FColor BoxColor = FColor::Blue;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Appearance")
	FColor DisplayColor = FColor::White;

	UPROPERTY(EditAnywhere, BlueprintReadWrite, Category = "SRDisplay|Appearance", meta = (ClampMin = "1.0", ClampMax = "200.0", UIMin = "1.0", UIMax = "200.0"))
	float LineThickness = 1.f;

public:
	float GetRealToVirtualScale();

#if WITH_EDITOR
	void PostEditChangeProperty(struct FPropertyChangedEvent& e) override;
#endif

private:
	USRDisplayDrawBoxComponent* Box = nullptr;
	UStaticMeshComponent* StaticMeshComponent;
	UStaticMesh* StaticMesh;
};
