// Fill out your copyright notice in the Description page of Project Settings.


#include "InputManagerComponent.h"
#include <RawInput.h>
#include <Windows/WindowsApplication.h>
#include <RawInputFunctionLibrary.h>
#include <RawInputSettings.h>

DEFINE_LOG_CATEGORY(LogInputManagerComponent);

// Sets default values for this component's properties
UInputManagerComponent::UInputManagerComponent()
{
	PrimaryComponentTick.bCanEverTick = true;

	OnInputDeviceAttachedHandler.Clear();
	DeviceIdList.Reset();
	const URawInputSettings* RawInputSettings = GetDefault<URawInputSettings>();
	for (const FRawInputDeviceConfiguration& DeviceConfig : RawInputSettings->DeviceConfigurations)
	{
		const int32 VendorID = FCString::Strtoi(*DeviceConfig.VendorID, nullptr, 16);
		const int32 ProductID = FCString::Strtoi(*DeviceConfig.ProductID, nullptr, 16);

		DeviceIdList.Add(DeviceId { .VendorID = VendorID, .ProductID = ProductID });
	}
}


// Called when the game starts
void UInputManagerComponent::BeginPlay()
{
	Super::BeginPlay();

	IPlatformInputDeviceMapper::Get().GetOnInputDeviceConnectionChange().AddUObject(this, &UInputManagerComponent::OnInputDeviceConnectionChange);

	IsBuiltInDeviceAttached = FSlateApplication::Get().IsGamepadAttached();
	LastDeviceAttached = IsDeviceAttached();
}

void UInputManagerComponent::BeginDestroy()
{
	Super::BeginDestroy();

	OnInputDeviceAttachedHandler.Clear();

	IRawInput* RawInput = static_cast<FRawInputPlugin*>(&FRawInputPlugin::Get())->GetRawInputDevice().Get();
	if (DeviceHandle != INDEX_NONE)
	{
		RawInput->RemoveRegisteredInputDevice(DeviceHandle);
		DeviceHandle = INDEX_NONE;
	}
}

// Called every frame
void UInputManagerComponent::TickComponent(float DeltaTime, ELevelTick TickType, FActorComponentTickFunction* ThisTickFunction)
{
	Super::TickComponent(DeltaTime, TickType, ThisTickFunction);

	if (DeviceHandle == INDEX_NONE)
	{
		RegisterDevice();
	}
	else 
	{
		IRawInput* RawInput = static_cast<FRawInputPlugin*>(&FRawInputPlugin::Get())->GetRawInputDevice().Get();
		RawInput->QueryConnectedDevices();
	}

	UpdateRawInputDeviceAttached();
	
	const bool isAttached = IsDeviceAttached();
	if (LastDeviceAttached != isAttached)
	{
		OnInputDeviceAttachedHandler.Broadcast(isAttached);
		LastDeviceAttached = isAttached;
	}
}

void UInputManagerComponent::OnInputDeviceConnectionChange(EInputDeviceConnectionState NewConnectionState, FPlatformUserId PlatformUserId, FInputDeviceId InputDeviceId)
{
	IsBuiltInDeviceAttached = FSlateApplication::Get().IsGamepadAttached();
}

void UInputManagerComponent::RegisterDevice()
{
	IRawInput* RawInput = static_cast<FRawInputPlugin*>(&FRawInputPlugin::Get())->GetRawInputDevice().Get();

	const uint32 Flags = 0;
	const int32 PageID = 0x01;
	const int32 DeviceID = 0x05;
	DeviceHandle = RawInput->RegisterInputDevice(RIM_TYPEHID, Flags, DeviceID, PageID, nullptr);
}

void UInputManagerComponent::UpdateRawInputDeviceAttached()
{
	auto RegisteredDevices = URawInputFunctionLibrary::GetRegisteredDevices();
	for (const auto& RegisteredDevice : RegisteredDevices)
	{
		for (const auto& deviceId : DeviceIdList) 
		{
			if (deviceId.VendorID == RegisteredDevice.VendorID && deviceId.ProductID == RegisteredDevice.ProductID)
			{
				IsRawInputDeviceAttached = true;
				return;
			}
		}
	}

	IsRawInputDeviceAttached = false;
}
