/*
 * Copyright 2023 Sony Corporation
 */

#pragma once
#include "Window/SRDisplaySample2DWindow.h"
#include "Window/SRDisplaySample2DSceneViewport.h"
#include "Blueprint/SRDisplayFunctionLibrary.h"

// <-- Engine Header -->
#include <Engine/Engine.h>
#include <Engine/GameViewportClient.h>
#include <UnrealEngine.h>
#include <Framework/Application/SlateApplication.h>
#include <Windows/WindowsHWrapper.h>

TSharedPtr<FSample2DWindow> FSample2DWindow::PlayWindow = nullptr;

FDelegateHandle RenderTextureDelegateHundle;

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 5
FDelegateHandle SampleWindowRenderTextureDelegateHandle;
#endif

FSample2DWindow::FSample2DWindow(FVector2D Resolution)
{
#if PLATFORM_WINDOWS
	HWND WindowHandle = (HWND)GEngine->GameViewport->GetWindow()->GetNativeWindow()->GetOSWindowHandle();
	if (WindowHandle)
	{
		::EnableWindow(WindowHandle, false);
		::SetParent(WindowHandle, HWND_MESSAGE);
	}
#endif

	Window = SNew(SWindow)
		.Type(EWindowType::GameWindow)
		.Title(FText::FromString("SRDisplay Sample Window"))
		.AutoCenter(EAutoCenter::PrimaryWorkArea)
		.ScreenPosition(FVector2D(0, 0))
		.ClientSize(Resolution)
		.SizingRule(ESizingRule::UserSized)
		.IsTopmostWindow(true)
		.ActivationPolicy(EWindowActivationPolicy::FirstShown)
		.FocusWhenFirstShown(true)
		.UseOSWindowBorder(true)
		.IsInitiallyMinimized(true)
		.SaneWindowPlacement(false);

	InDelegate.BindRaw(this, &FSample2DWindow::OnWindowClosed);
	Window->SetOnWindowClosed(InDelegate);

	PlaySceneViewport = SNew(SSample2DSceneViewport);
	PlaySceneViewport->Init(Resolution.X, Resolution.Y);
	Window->SetContent(PlaySceneViewport.ToSharedRef());
}

FSample2DWindow::~FSample2DWindow()
{
	InDelegate.Unbind();
}

TSharedPtr<SWindow> FSample2DWindow::GetWindow()
{
	if (Window.IsValid())
	{
		return Window;
	}
	return nullptr;
}

UTextureRenderTarget2D* FSample2DWindow::GetRenderTarget()
{
	if (PlaySceneViewport.IsValid())
	{
		return PlaySceneViewport->GetRenderTarget();
	}
	return nullptr;
}

void FSample2DWindow::RenderTexture_RenderThread(FRHICommandListImmediate& RHICmdList)
{
	if (PlaySceneViewport.IsValid())
	{
		PlaySceneViewport->RenderTexture_RenderThread(RHICmdList);
	}
}
#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 5
void FSample2DWindow::RenderTexture_RenderThread(FRDGBuilder& GraphBuilder)
{
	if (PlaySceneViewport.IsValid())
	{
		PlaySceneViewport->RenderTexture_RenderThread(GraphBuilder);
	}
}
#endif

void FSample2DWindow::Initialize(FVector2D Resolution, FIntRect ScreenPosition)
{
	if (!PlayWindow.IsValid())
	{
		FSample2DWindow::PlayWindow = MakeShareable(new FSample2DWindow(Resolution));
	}

	if (GEngine && FSample2DWindow::PlayWindow && FSample2DWindow::PlayWindow->GetWindow())
	{
		const float WindowPosX = static_cast<float>(ScreenPosition.Min.X);
		const float WindowPosY = static_cast<float>(ScreenPosition.Min.Y);
		const FVector2D Position(WindowPosX, WindowPosY);
		const float WindowWidth = static_cast<float>(ScreenPosition.Width());
		const float WindowHeight = static_cast<float>(ScreenPosition.Height());
		const FVector2D Size(WindowWidth, WindowHeight);

		FSample2DWindow::GetPlayWindow()->GetWindow()->MoveWindowTo(Position);
		FSample2DWindow::GetPlayWindow()->GetWindow()->Resize(Size);

		FSlateApplication::Get().AddWindowAsNativeChild(
			FSample2DWindow::GetPlayWindow()->GetWindow().ToSharedRef(), GEngine->GameViewport->GetWindow().ToSharedRef());
		FSample2DWindow::GetPlayWindow()->GetWindow()->SetWindowMode(EWindowMode::Windowed);
	}

	if (GEngine)
	{
		UGameUserSettings* UserSettings = GEngine->GetGameUserSettings();
		UserSettings->SetFullscreenMode(EWindowMode::Windowed);
		UserSettings->ApplySettings(false);
		UserSettings->SetFullscreenMode(EWindowMode::WindowedFullscreen);
		UserSettings->ApplySettings(false);
	}
	FSample2DWindow::RegisterDelegate();
}

void FSample2DWindow::RegisterDelegate()
{
	srdisplay_module::FSRDisplaySystem* SRDisplaySystem = USRDisplayFunctionLibrary::GetSRDisplaySystem();
	FOnSRDRenderTextureCompletedDelegate& OnSRDRenderTextureCompletedDelegate = SRDisplaySystem->GetSRDRenderTextureCompletedDelegate();
	RenderTextureDelegateHundle = OnSRDRenderTextureCompletedDelegate.AddRaw(FSample2DWindow::GetPlayWindow(), &FSample2DWindow::RenderingSRDisplay2DWindow);

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 5
	FSlateRenderer* Renderer = FSlateApplication::Get().GetRenderer();
	SampleWindowRenderTextureDelegateHandle = Renderer->OnAddBackBufferReadyToPresentPass().AddRaw(FSample2DWindow::GetPlayWindow(), &FSample2DWindow::RenderingSRDisplay2DWindow_RenderTread);
#endif
}

void FSample2DWindow::DeleteDelegate()
{
	srdisplay_module::FSRDisplaySystem* SRDisplaySystem = USRDisplayFunctionLibrary::GetSRDisplaySystem();
	SRDisplaySystem->GetSRDRenderTextureCompletedDelegate().Remove(RenderTextureDelegateHundle);

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 5
	FSlateRenderer* Renderer = FSlateApplication::Get().GetRenderer();
	Renderer->OnAddBackBufferReadyToPresentPass().Remove(SampleWindowRenderTextureDelegateHandle);
#endif
}

void FSample2DWindow::Finalize()
{
	FSample2DWindow::DeleteDelegate();
	if (FSample2DWindow::PlayWindow.IsValid())
	{
		if (FSample2DWindow::PlayWindow->PlaySceneViewport.IsValid())
		{
			FSample2DWindow::PlayWindow->PlaySceneViewport->Shutdown();
			FSample2DWindow::PlayWindow->PlaySceneViewport.Reset();
			FSample2DWindow::PlayWindow->PlaySceneViewport = nullptr;
			FSlateApplication::Get().RequestDestroyWindow(FSample2DWindow::GetPlayWindow()->GetWindow().ToSharedRef());
		}
		
		FSample2DWindow::PlayWindow.Reset();
		FSample2DWindow::PlayWindow = nullptr;
	}
}

FSample2DWindow* FSample2DWindow::GetPlayWindow()
{
	if (PlayWindow.IsValid())
	{
		return PlayWindow.Get();
	}
	return nullptr;
}

void FSample2DWindow::OnWindowClosed(const TSharedRef<SWindow>& InWindow)
{
	FSample2DWindow::Finalize();
}

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
void FSample2DWindow::RenderingSRDisplay2DWindow(FRHICommandListImmediate& RHICmdList, FTextureRHIRef SrcTexture)
#else
void FSample2DWindow::RenderingSRDisplay2DWindow(FRHICommandListImmediate& RHICmdList, FTexture2DRHIRef SrcTexture)
#endif
{
	const FVector2D WindowSize = FVector2D(960, 540);
	CopySRDisplayTexture(RHICmdList, FSample2DWindow::GetPlayWindow()->GetRenderTarget()->GetRenderTargetResource()->GetRenderTargetTexture(), SrcTexture, WindowSize);
	FSample2DWindow::GetPlayWindow()->RenderTexture_RenderThread(RHICmdList);
}

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 5
void FSample2DWindow::RenderingSRDisplay2DWindow_RenderTread(FRDGBuilder& GraphBuilder, SWindow& window, FRDGTexture* Texture)
{
	FSample2DWindow::GetPlayWindow()->RenderTexture_RenderThread(GraphBuilder);
}
#endif

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 1
void FSample2DWindow::CopySRDisplayTexture(FRHICommandListImmediate& RHICmdList, FRHITexture* TargetTexture, FRHITexture* SrcTexture, FVector2D WindowSize) const
#else
void FSample2DWindow::CopySRDisplayTexture(FRHICommandListImmediate& RHICmdList, FRHITexture2D* TargetTexture, FRHITexture2D* SrcTexture, FVector2D WindowSize) const
#endif

{
	const FVector2D ViewportSize = FVector2D(WindowSize.X, WindowSize.Y);
	const FVector2D TextureSize = FVector2D(SrcTexture->GetSizeX()/2, SrcTexture->GetSizeY());

	FRHIRenderPassInfo RPInfoTemp(TargetTexture, ERenderTargetActions::Load_Store);
	RHICmdList.BeginRenderPass(RPInfoTemp, TEXT("CopySRDisplayTexture"));
	{
		DrawClearQuad(RHICmdList, FLinearColor(0.0f, 0.0f, 0.0f, 1.0f));
		RHICmdList.SetViewport(0, 0, 0.0f, ViewportSize.X, ViewportSize.Y, 1.0f);

		FGraphicsPipelineStateInitializer GraphicsPSOInit;
		RHICmdList.ApplyCachedRenderTargets(GraphicsPSOInit);

		const auto FeatureLevel = GMaxRHIFeatureLevel;
		auto ShaderMap = GetGlobalShaderMap(FeatureLevel);
		TShaderMapRef<FScreenVS> VertexShader(ShaderMap);
		TShaderMapRef<FScreenPS> PixelShader(ShaderMap);

		GraphicsPSOInit.BlendState = TStaticBlendState<>::GetRHI();
		GraphicsPSOInit.RasterizerState = TStaticRasterizerState<>::GetRHI();
		GraphicsPSOInit.DepthStencilState = TStaticDepthStencilState<false, CF_Always>::GetRHI();
		GraphicsPSOInit.PrimitiveType = PT_TriangleList;
		GraphicsPSOInit.BoundShaderState.VertexDeclarationRHI = GFilterVertexDeclaration.VertexDeclarationRHI;
		GraphicsPSOInit.BoundShaderState.VertexShaderRHI = VertexShader.GetVertexShader();
		GraphicsPSOInit.BoundShaderState.PixelShaderRHI = PixelShader.GetPixelShader();

#if ENGINE_MAJOR_VERSION == 5 && ENGINE_MINOR_VERSION >= 5
		SetGraphicsPipelineState(RHICmdList, GraphicsPSOInit, 0);
		FRHIBatchedShaderParameters& BatchedParameters = RHICmdList.GetScratchShaderParameters();
		PixelShader->SetParameters(BatchedParameters, TStaticSamplerState<SF_Bilinear>::GetRHI(), SrcTexture);
		RHICmdList.SetBatchedShaderParameters(PixelShader.GetPixelShader(), BatchedParameters);
#else
		SetGraphicsPipelineState(RHICmdList, GraphicsPSOInit);
		PixelShader->SetParameters(RHICmdList, TStaticSamplerState<SF_Bilinear>::GetRHI(), SrcTexture);
#endif

		static const FName RendererModuleName("Renderer");
		IRendererModule* RendererModule = FModuleManager::GetModulePtr<IRendererModule>(RendererModuleName);

		RendererModule->DrawRectangle(
			RHICmdList,
			0.f, 0.f,
			ViewportSize.X, ViewportSize.Y,
			0.f, 0.f,
			0.5f, 1.f,
			FIntPoint(ViewportSize.X, ViewportSize.Y),
			FIntPoint(1, 1),
			VertexShader,
			EDRF_Default);
	}
	RHICmdList.EndRenderPass();
}