#include <iostream>
#include <cstdint>
#include <iterator>
#include <string>
#include <memory>
#include <vector>
#include <chrono>

#include "GLManager.h"
#include "QuadCopyShader.h"
#include "HomographyCopyShader.h"

#include <GL/gl3w.h>
#include <GLFW/glfw3.h>
#define GLFW_EXPOSE_NATIVE_WIN32
#include <GLFW/glfw3native.h>

#include <glm/glm.hpp>
#include <glm/gtc/quaternion.hpp>
#include <glm/gtc/matrix_transform.hpp>

#include <xr_api_wrapper.h>
#include <xr_api_wrapper_utility.h>
#include <xr_basic_api_wrapper.h>
#include <xr_extension_api_wrapper.h>

using namespace sony::oz::xr_runtime;
bool wallmount_enabled = false;

class SampleApp
{
public:
	explicit SampleApp(GLFWwindow* window);
	~SampleApp();

	SampleApp(const SampleApp&) = delete;
	SampleApp& operator=(const SampleApp&) = delete;

	bool Initialize();
	void Terminate();
	void Run();

private:

	void SetupScene();
	void SetupQuad();
	bool SetupCopyShader();
	bool SetupHomographyShader();
	void DestoryScene();
	void UpdateScene();
	void RenderScene();
	void CloseApp();
	void CopyQuad(const FrameBuffer& target, GLuint texture, const Quad& quad);
	void CopyQuad(GLuint fbo, uint32_t w, uint32_t h, GLuint texture, const Quad& quad);
	void CopyHomography(const FrameBuffer& target, GLuint texture, const Quad& quad, const glm::mat3& homographyMtx, const glm::mat3& invHomographyMtx);
	void CheckMouseInput();
	void CheckKeyboardInput();

	bool OnSRDEnable();
	void OnSRDDisable();
	void OnSetupSRDWindow();
	void OnSetupSRDFramebuffer();


private:

	// GLFW
	GLFWwindow* m_window = nullptr;

	// SRD
	bool m_enableSRD = true;
	SonyOzSessionHandle	m_sessionHandle = nullptr;
	SonyOzRect m_srdRect = {};
	uint32_t	m_deviceWidth = 0;
	uint32_t	m_deviceHeight = 0;
	SonyOzDisplaySpec	m_srdSpec;
	float	m_srdScreenWidthCm = 0;
	float	m_srdScreenHeightCm = 0;
	float	m_srdHeightCm = 0;
	float	m_srdDepthCm = 0;
	float	m_planeWidth = 16;
	float	m_planeHeight = 9;


	std::shared_ptr<FrameBuffer>	m_leftTarget;
	std::shared_ptr<FrameBuffer>	m_rightTarget;
	std::shared_ptr<FrameBuffer>	m_leftHmTarget;
	std::shared_ptr<FrameBuffer>	m_rightHmTarget;
	std::shared_ptr<FrameBuffer>	m_sideBySideTarget;
	std::shared_ptr<FrameBuffer>	m_compositeTarget;

	// Draw
	GLManager m_glManager;
	SceneInfo	m_sceneInfo;
	std::shared_ptr<DrawObject>	m_backPlane;
	std::shared_ptr<DrawObject>	m_bottomPlane;
	std::shared_ptr<DrawObject>	m_sampleObj;
	std::shared_ptr<DrawObject> m_mousePointerObj;

	std::shared_ptr<ShaderProgram>	m_copyShader;
	std::shared_ptr<ShaderProgram>	m_homographyShader;
	GLuint	m_quadVBO = 0;
	GLuint	m_quadIBO = 0;
	GLuint	m_copyQuadVAO = 0;
	GLuint	m_homographyQuadVAO = 0;
	GLint m_copy_uTex = -1;
	GLint m_homography_copy_uMtx = -1;
	GLint m_homography_copy_uInvMtx = -1;
	GLint m_homography_copy_uTex = -1;

	std::chrono::high_resolution_clock::time_point m_beforeUpdateTime;

	bool m_bIsHighImageQualitySupported = false;
	bool m_bIsPerformancePrioritySupported = false;
	bool m_bIsWallmountSupported = false;
};

SampleApp::SampleApp(GLFWwindow* window)
	: m_window(window)
{
}

SampleApp::~SampleApp()
{
	Terminate();
}

bool SampleApp::Initialize()
{
	// Hide the system cursor
	glfwSetInputMode(m_window, GLFW_CURSOR, GLFW_CURSOR_HIDDEN);

	if (!m_glManager.Setup())
	{
		return false;
	}

	if (m_enableSRD)
	{
		if (!OnSRDEnable())
		{
			return false;
		}

		OnSetupSRDWindow();

		OnSetupSRDFramebuffer();
	}

	SetupScene();

	SetupQuad();

	if (!SetupCopyShader())
	{
		return false;
	}

	if (!m_bIsHighImageQualitySupported) {
		if (!SetupHomographyShader())
		{
			return false;
		}
	}
	m_beforeUpdateTime = std::chrono::high_resolution_clock::now();

	return true;
}

void SampleApp::Terminate()
{
	if (m_copyQuadVAO != 0)
	{
		glDeleteVertexArrays(1, &m_copyQuadVAO);
		m_copyQuadVAO = 0;
	}

	if (m_homographyQuadVAO != 0)
	{
		glDeleteVertexArrays(1, &m_homographyQuadVAO);
		m_homographyQuadVAO = 0;
	}

	if (m_quadIBO != 0)
	{
		glDeleteBuffers(1, &m_quadIBO);
		m_quadIBO = 0;
	}

	if (m_quadVBO != 0)
	{
		glDeleteBuffers(1, &m_quadVBO);
		m_quadVBO = 0;
	}

	OnSRDDisable();
}

static glm::mat4 MakeViewMatrix(const SonyOzPosef& pose)
{
	const auto inv_xz = glm::mat3(
		-1, 0, 0,
		0, 1, 0,
		0, 0, -1
	);

	auto camRot = glm::quat(pose.orientation.w, pose.orientation.x, pose.orientation.y, pose.orientation.z);
	auto camPos = glm::vec3(pose.position.x, pose.position.y, pose.position.z);
	camPos = (glm::mat4(inv_xz) * glm::translate(glm::mat4(1), camPos) * glm::mat4(inv_xz))[3];
	camRot = glm::quat_cast(inv_xz * glm::mat3_cast(camRot) * inv_xz);

	// Convert units from meters to centimeters.
	camPos = camPos * 100.0f;
	const auto view = glm::inverse(glm::translate(glm::mat4(1), camPos) * glm::mat4_cast(camRot));

	return view;
}

static glm::mat4 MakeProjectionMatrix(const SonyOzProjection& proj, float nearClip, float farClip)
{
	const float left = nearClip * tanf(proj.half_angles_left);
	const float right = nearClip * tanf(proj.half_angles_right);
	const float top = nearClip * tanf(proj.half_angles_top);
	const float bottom = nearClip * tanf(proj.half_angles_bottom);
	return glm::frustumRH(left, right, bottom, top, nearClip, farClip);
}

static glm::vec2 WorldToViewport(const glm::mat4& vp, const glm::vec3& pos)
{
	glm::vec4 screenPos = vp * glm::vec4(pos, 1);
	screenPos /= screenPos.w;
	return glm::vec2((screenPos.x + 1.0f) * 0.5f, 1.0f - (screenPos.y + 1.0f) * 0.5f);
}

static glm::mat3 MakeHomographyMatrix(const glm::vec2 viewportPoints[4])
{
	const auto p00 = viewportPoints[0];
	const auto p01 = viewportPoints[1];
	const auto p10 = viewportPoints[2];
	const auto p11 = viewportPoints[3];

	const auto x00 = p00.x;
	const auto y00 = p00.y;
	const auto x01 = p01.x;
	const auto y01 = p01.y;
	const auto x10 = p10.x;
	const auto y10 = p10.y;
	const auto x11 = p11.x;
	const auto y11 = p11.y;

	const auto a = x10 - x11;
	const auto b = x01 - x11;
	const auto c = x00 - x01 - x10 + x11;
	const auto d = y10 - y11;
	const auto e = y01 - y11;
	const auto f = y00 - y01 - y10 + y11;

	const auto h13 = x00;
	const auto h23 = y00;
	const auto h32 = (c * d - a * f) / (b * d - a * e);
	const auto h31 = (c * e - b * f) / (a * e - b * d);
	const auto h11 = x10 - x00 + h31 * x10;
	const auto h12 = x01 - x00 + h32 * x01;
	const auto h21 = y10 - y00 + h31 * y10;
	const auto h22 = y01 - y00 + h32 * y01;

	return glm::mat3(h11, h12, h13, h21, h22, h23, h31, h32, 1.0f);
}

static glm::mat3 MakeInverseHomographyMatrix(const glm::mat3& m)
{
	return glm::inverse(m);
}

void SampleApp::Run()
{
	int w, h;
	glfwGetFramebufferSize(m_window, &w, &h);

	if (m_bIsPerformancePrioritySupported) {
		w /= 2;
		h /= 2;
	}
	glViewport(0, 0, w, h);

	m_sceneInfo.m_lightColor[0] = glm::vec3(1);
	m_sceneInfo.m_lightColor[1] = glm::vec3(1);
	m_sceneInfo.m_lightColor[2] = glm::vec3(0);
	m_sceneInfo.m_lightColor[3] = glm::vec3(0);
	m_sceneInfo.m_lightDir[0] = glm::vec3(0, -1, 0);
	m_sceneInfo.m_lightDir[1] = glm::vec3(-1, -1, -1);
	m_sceneInfo.m_lightDir[2] = glm::vec3(1, 1, 1);
	m_sceneInfo.m_lightDir[3] = glm::vec3(0, 0, -1);

	UpdateScene();
	CheckMouseInput();
	CheckKeyboardInput();

	if (m_enableSRD)
	{
		SonyOzSessionState state;
		SonyOzResult result = SonyOzResult::SUCCESS;
		if (GetSessionState(m_sessionHandle, &state) != SonyOzResult::SUCCESS)
		{
			return;
		}
		if (state != SonyOzSessionState::RUNNING)
		{
			return;
		}

		// Get the Pose for each camera.
		SonyOzPosef head_pose, left_pose, right_pose;
		bool headValid, leftValid, rightValid;
		UpdateTrackingResultCache(m_sessionHandle);
		SonyOzResult headResult = GetCachedPose(m_sessionHandle, SonyOzPoseId::HEAD, &head_pose, &headValid);
		SonyOzResult leftResult = GetCachedPose(m_sessionHandle, SonyOzPoseId::LEFT_EYE, &left_pose, &leftValid);
		SonyOzResult rightResult = GetCachedPose(m_sessionHandle, SonyOzPoseId::RIGHT_EYE, &right_pose, &rightValid);
		if ((headResult != SonyOzResult::SUCCESS) || (leftResult != SonyOzResult::SUCCESS) || (rightResult != SonyOzResult::SUCCESS))
		{
			std::cout << "GetCachedPose Error." << std::endl;
			CloseApp();
			return;
		}

		// if wallmount mode is enabled, pose needs to be converted based on 90 degrees.
		if (m_bIsWallmountSupported) {
			head_pose.position = SonyOzVector3f  {head_pose.position.x, 
												  head_pose.position.y* cos(PI / 4) + head_pose.position.z * sin(PI / 4),
												  head_pose.position.y * -sin(PI / 4) + head_pose.position.z * cos(PI / 4)};
			left_pose.position = SonyOzVector3f  {left_pose.position.x,
												  left_pose.position.y * cos(PI / 4) + left_pose.position.z * sin(PI / 4),
												  left_pose.position.y * -sin(PI / 4) + left_pose.position.z * cos(PI / 4)};
			right_pose.position = SonyOzVector3f {right_pose.position.x,
												  right_pose.position.y * cos(PI / 4) + right_pose.position.z * sin(PI / 4),
												  right_pose.position.y * -sin(PI / 4) + right_pose.position.z * cos(PI / 4)};

			head_pose.orientation.x = left_pose.orientation.x = right_pose.orientation.x = 0;
		}

		//const auto view = MakeViewMatrix(head_pose);
		const auto viewL = MakeViewMatrix(left_pose);
		const auto viewR = MakeViewMatrix(right_pose);

		// Get the Projection for each camera.
		SonyOzProjection head_projection, left_eye_projection, right_eye_projection;
		headResult = GetProjection(m_sessionHandle, SonyOzPoseId::HEAD, &head_projection);
		leftResult = GetProjection(m_sessionHandle, SonyOzPoseId::LEFT_EYE, &left_eye_projection);
		rightResult = GetProjection(m_sessionHandle, SonyOzPoseId::RIGHT_EYE, &right_eye_projection);
		if ((headResult != SonyOzResult::SUCCESS) || (leftResult != SonyOzResult::SUCCESS) || (rightResult != SonyOzResult::SUCCESS))
		{
			std::cout << "GetProjection Error." << std::endl;
			CloseApp();
			return;
		}

		//const auto proj = MakeProjectionMatrix(head_projection, 1.0f, 1000.0f);
		const auto projL = MakeProjectionMatrix(left_eye_projection, 1.0f, 1000.0f);
		const auto projR = MakeProjectionMatrix(right_eye_projection, 1.0f, 1000.0f);

		// Render left view
		m_sceneInfo.m_view = viewL;
		m_sceneInfo.m_projection = projL;
		glBindFramebuffer(GL_FRAMEBUFFER, m_leftTarget->m_fbo);
		RenderScene();

		// Render right view
		m_sceneInfo.m_view = viewR;
		m_sceneInfo.m_projection = projR;
		glBindFramebuffer(GL_FRAMEBUFFER, m_rightTarget->m_fbo);
		RenderScene();

		glBindFramebuffer(GL_FRAMEBUFFER, 0);

		if (!m_bIsHighImageQualitySupported) {
			// Homography
			glm::vec3 srdDispalyPlanePositions[4];
			{
				const float x0 = -m_srdScreenWidthCm * 0.5f;	// Left
				const float x1 = +m_srdScreenWidthCm * 0.5f;	// Right
				const float y0 = m_srdHeightCm;					// Top
				const float y1 = 0.0f;							// Bottom
				const float z0 = 0.0f;							// Front
				const float z1 = -m_srdDepthCm;					// Back
				srdDispalyPlanePositions[0] = glm::vec3(x0, y0, z1);	// Left Top
				srdDispalyPlanePositions[1] = glm::vec3(x0, y1, z0);	// Left Bottom
				srdDispalyPlanePositions[2] = glm::vec3(x1, y0, z1);	// Right Top
				srdDispalyPlanePositions[3] = glm::vec3(x1, y1, z0);	// Right Bottom
			}

			// Left eye
			const auto vpLeft = projL * viewL;
			const glm::vec2 viewportLeft[] =
			{
				WorldToViewport(vpLeft, srdDispalyPlanePositions[0]),
				WorldToViewport(vpLeft, srdDispalyPlanePositions[1]),
				WorldToViewport(vpLeft, srdDispalyPlanePositions[2]),
				WorldToViewport(vpLeft, srdDispalyPlanePositions[3]),
			};
			glm::mat3 homographyL = MakeHomographyMatrix(viewportLeft);
			glm::mat3 inverseHomographyL = MakeInverseHomographyMatrix(homographyL);
			Quad hmLQuad;
			hmLQuad.m_quad[0].m_position = glm::vec2(0, 0);
			hmLQuad.m_quad[1].m_position = glm::vec2(0, 1);
			hmLQuad.m_quad[2].m_position = glm::vec2(1, 0);
			hmLQuad.m_quad[3].m_position = glm::vec2(1, 1);
			CopyHomography(*m_leftHmTarget, m_leftTarget->m_texture, hmLQuad, inverseHomographyL, homographyL);


			// Right eye
			const auto vpRight = projR * viewR;
			const glm::vec2 viewportRight[] =
			{
				WorldToViewport(vpRight, srdDispalyPlanePositions[0]),
				WorldToViewport(vpRight, srdDispalyPlanePositions[1]),
				WorldToViewport(vpRight, srdDispalyPlanePositions[2]),
				WorldToViewport(vpRight, srdDispalyPlanePositions[3]),
			};
			glm::mat3 homographyR = MakeHomographyMatrix(viewportRight);
			glm::mat3 inverseHomographyR = MakeInverseHomographyMatrix(homographyR);
			Quad hmRQuad;
			hmRQuad.m_quad[0].m_position = glm::vec2(0, 0);
			hmRQuad.m_quad[1].m_position = glm::vec2(0, 1);
			hmRQuad.m_quad[2].m_position = glm::vec2(1, 0);
			hmRQuad.m_quad[3].m_position = glm::vec2(1, 1);
			CopyHomography(*m_rightHmTarget, m_rightTarget->m_texture, hmRQuad, inverseHomographyR, homographyR);
		}


		Quad copyLeftQuad;
		copyLeftQuad.m_quad[2].m_position.x = 0;
		copyLeftQuad.m_quad[3].m_position.x = 0;
		if (m_bIsHighImageQualitySupported) {
			CopyQuad(*m_sideBySideTarget, m_leftTarget->m_texture, copyLeftQuad);
		}
		else{
			CopyQuad(*m_sideBySideTarget, m_leftHmTarget->m_texture, copyLeftQuad);
		}

		Quad copyRightQuad;
		copyRightQuad.m_quad[0].m_position.x = 0;
		copyRightQuad.m_quad[1].m_position.x = 0;
		if (m_bIsHighImageQualitySupported) {
			CopyQuad(*m_sideBySideTarget, m_rightTarget->m_texture, copyRightQuad);
		}
		else {
			CopyQuad(*m_sideBySideTarget, m_rightHmTarget->m_texture, copyRightQuad);
		}

		result = SubmitOpengl(m_sessionHandle, m_sideBySideTarget->m_texture, false, m_compositeTarget->m_texture);

		Quad copyCompositeQuad;
		CopyQuad(0, m_deviceWidth, m_deviceHeight, m_compositeTarget->m_texture, copyCompositeQuad);
	}
	else
	{
		float aspect = float(w) / float(h);
		m_sceneInfo.m_view = glm::lookAt(glm::vec3(0, 12, 20), glm::vec3(0, 0, 0), glm::vec3(0, 1, 0));
		m_sceneInfo.m_projection = glm::perspective(glm::radians(30.0f), aspect, 0.1f, 100.0f);
		RenderScene();
	}
}

void SampleApp::SetupScene()
{
	glm::mat4 srdCenter = glm::mat4(1);
	if (m_enableSRD)
	{
		m_planeWidth = m_srdScreenWidthCm;
		m_planeHeight = m_srdHeightCm;
		if (m_bIsWallmountSupported) {
			srdCenter = glm::translate(glm::mat4(1), glm::vec3(0, m_srdHeightCm * 0.5f, -m_srdScreenHeightCm * cos(PI / 4) * 0.5f));
		}
		else {
			srdCenter = glm::translate(glm::mat4(1), glm::vec3(0, m_srdHeightCm * 0.5f, -m_srdDepthCm * 0.5f));
		}
	}

	m_backPlane = m_glManager.CreatePlane(m_planeWidth, m_planeHeight);
	m_backPlane->m_transform = glm::translate(srdCenter, glm::vec3(0, 0, -m_planeHeight / 2));
	m_backPlane->m_diffuse = glm::vec4(255.0f / 255.0f, 250.0f / 255.0f, 240.0f / 255.0f, 1.0f);

	m_bottomPlane = m_glManager.CreatePlane(m_planeWidth, m_planeHeight);
	m_bottomPlane->m_transform = glm::translate(srdCenter, glm::vec3(0, -m_planeHeight / 2, 0));
	m_bottomPlane->m_transform = glm::rotate(m_bottomPlane->m_transform, glm::radians(-90.0f), glm::vec3(1, 0, 0));
	m_bottomPlane->m_diffuse = glm::vec4(255.0f / 255.0f, 250.0f / 255.0f, 240.0f / 255.0f, 1.0f);

	m_sampleObj = m_glManager.CreateCube(m_planeHeight / 4.0f);
	m_sampleObj->m_transform = srdCenter;
	m_sampleObj->m_diffuse = glm::vec4(30.0f / 255.0f, 144.0f / 255.0f, 255.0f / 255.0f, 1.0f);

	m_mousePointerObj = m_glManager.CreateSphere(.5f);
	m_mousePointerObj->m_transform = srdCenter;
	m_mousePointerObj->m_diffuse = glm::vec4(170.0f / 255.0f, 170.0f / 255.0f, 170.0f / 255.0f, 1.0f);
}

void SampleApp::SetupQuad()
{
	glGenBuffers(1, &m_quadVBO);
	glBindBuffer(GL_ARRAY_BUFFER, m_quadVBO);
	glBufferData(GL_ARRAY_BUFFER, sizeof(Quad), nullptr, GL_DYNAMIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	uint8_t indices[] = {
		0, 1, 2,
		1, 3, 2,
	};
	glGenBuffers(1, &m_quadIBO);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_quadIBO);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, std::size(indices) * sizeof(uint8_t), &indices[0], GL_STATIC_DRAW);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
}

bool SampleApp::SetupCopyShader()
{
	m_copyShader = m_glManager.CreateShaderProgram(QuadCopyVertexShader, QuadCopyFragmentShader);
	if (m_copyShader == nullptr)
	{
		return false;
	}

	GLint inPos = glGetAttribLocation(m_copyShader->m_prog, "in_Pos");
	GLint inUV = glGetAttribLocation(m_copyShader->m_prog, "in_UV");

	m_copy_uTex = glGetUniformLocation(m_copyShader->m_prog, "u_Tex");

	glGenVertexArrays(1, &m_copyQuadVAO);
	glBindVertexArray(m_copyQuadVAO);

	glBindBuffer(GL_ARRAY_BUFFER, m_quadVBO);

	glVertexAttribPointer(inPos, 2, GL_FLOAT, GL_FALSE, sizeof(QuadVertex), (const void*)0);
	glEnableVertexAttribArray(inPos);

	glVertexAttribPointer(inUV, 2, GL_FLOAT, GL_FALSE, sizeof(QuadVertex), (const void*)(sizeof(glm::vec2)));
	glEnableVertexAttribArray(inUV);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_quadIBO);

	glBindVertexArray(0);

	return true;
}

bool SampleApp::SetupHomographyShader()
{
	m_homographyShader = m_glManager.CreateShaderProgram(HomographyCopyVertexShader, HomographyCopyFragmentShader);
	if (m_homographyShader == nullptr)
	{
		return false;
	}

	GLint inPos = glGetAttribLocation(m_homographyShader->m_prog, "in_Pos");

	m_homography_copy_uMtx = glGetUniformLocation(m_homographyShader->m_prog, "u_HomographyMatrix");
	m_homography_copy_uInvMtx = glGetUniformLocation(m_homographyShader->m_prog, "u_InvHomographyMatrix");
	m_homography_copy_uTex = glGetUniformLocation(m_homographyShader->m_prog, "u_Tex");

	glGenVertexArrays(1, &m_homographyQuadVAO);
	glBindVertexArray(m_homographyQuadVAO);

	glBindBuffer(GL_ARRAY_BUFFER, m_quadVBO);

	glVertexAttribPointer(inPos, 2, GL_FLOAT, GL_FALSE, sizeof(QuadVertex), (const void*)0);
	glEnableVertexAttribArray(inPos);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, m_quadIBO);

	glBindVertexArray(0);

	return true;
}

void SampleApp::DestoryScene()
{
}

void SampleApp::UpdateScene()
{
	auto time = std::chrono::high_resolution_clock::now();
	auto elapsed = std::chrono::duration_cast<std::chrono::microseconds>(time - m_beforeUpdateTime).count() / 1000.0f;
	auto r = 18.0f / elapsed;

	m_sampleObj->m_transform = glm::rotate(m_sampleObj->m_transform, glm::radians(r), glm::vec3(0, 1, 0));

	m_beforeUpdateTime = time;
}

void SampleApp::RenderScene()
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glEnable(GL_DEPTH_TEST);

	m_glManager.Draw(*m_backPlane, m_sceneInfo);
	m_glManager.Draw(*m_bottomPlane, m_sceneInfo);

	m_glManager.Draw(*m_sampleObj, m_sceneInfo);

	m_glManager.Draw(*m_mousePointerObj, m_sceneInfo);
}

void SampleApp::CloseApp()
{
	glfwSetWindowShouldClose(m_window, GLFW_TRUE);
}

void SampleApp::CopyQuad(const FrameBuffer& target, GLuint texture, const Quad& quad)
{
	CopyQuad(target.m_fbo, target.m_width, target.m_height, texture, quad);
}

void SampleApp::CopyQuad(GLuint fbo, uint32_t w, uint32_t h, GLuint texture, const Quad& quad)
{
	glBindFramebuffer(GL_FRAMEBUFFER, fbo);
	glViewport(0, 0, w, h);
	glDisable(GL_DEPTH_TEST);

	glUseProgram(m_copyShader->m_prog);
	glUniform1i(m_copy_uTex, 0);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, texture);

	glBindBuffer(GL_ARRAY_BUFFER, m_quadVBO);
	glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(Quad), &quad);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	glBindVertexArray(m_copyQuadVAO);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_BYTE, (const void*)0);
	glBindVertexArray(0);

	glUseProgram(0);

	glBindFramebuffer(GL_FRAMEBUFFER, 0);
}

void SampleApp::CopyHomography(const FrameBuffer& target, GLuint texture, const Quad& quad, const glm::mat3& homographyMtx, const glm::mat3& invHomographyMtx)
{
	glBindFramebuffer(GL_FRAMEBUFFER, target.m_fbo);
	glViewport(0, 0, m_deviceWidth, m_deviceHeight);
	glDisable(GL_DEPTH_TEST);
	glClear(GL_COLOR_BUFFER_BIT);

	glUseProgram(m_homographyShader->m_prog);
	glUniformMatrix3fv(m_homography_copy_uMtx, 1, GL_TRUE, &homographyMtx[0][0]);
	glUniformMatrix3fv(m_homography_copy_uInvMtx, 1, GL_TRUE, &invHomographyMtx[0][0]);
	glUniform1i(m_homography_copy_uTex, 0);

	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, texture);

	glBindBuffer(GL_ARRAY_BUFFER, m_quadVBO);
	glBufferSubData(GL_ARRAY_BUFFER, 0, sizeof(Quad), &quad);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	glBindVertexArray(m_copyQuadVAO);
	glDrawElements(GL_TRIANGLES, 6, GL_UNSIGNED_BYTE, (const void*)0);
	glBindVertexArray(0);

	glUseProgram(0);

	glBindFramebuffer(GL_FRAMEBUFFER, 0);
}

void SampleApp::CheckMouseInput()
{
	// Get mouse position in screen coordinates
	double xpos, ypos;
	glfwGetCursorPos(m_window, &xpos, &ypos);

	double xPercent = xpos / (m_deviceWidth - 1);
	double yPercent = ypos / (m_deviceHeight - 1);

	// Convert to position on the SRD plane in world coordinates
	double worldX = xPercent * m_srdScreenWidthCm - m_srdScreenWidthCm / 2;
	double worldY = m_srdHeightCm - yPercent * m_srdHeightCm;
	double worldZ = yPercent * m_srdDepthCm - m_srdDepthCm;

	m_mousePointerObj->m_transform = glm::translate(glm::mat4(1), glm::vec3(worldX, worldY, worldZ));

	// Change object color when left button is being pressed
	if (glfwGetMouseButton(m_window, GLFW_MOUSE_BUTTON_LEFT) == GLFW_PRESS)
	{
		m_mousePointerObj->m_diffuse = glm::vec4(170.0f / 255.0f, 85.0f / 255.0f, 85.0f / 255.0f, 1.0f);
	}
	else if (glfwGetMouseButton(m_window, GLFW_MOUSE_BUTTON_LEFT) == GLFW_RELEASE)
	{
		m_mousePointerObj->m_diffuse = glm::vec4(170.0f / 255.0f, 170.0f / 255.0f, 170.0f / 255.0f, 1.0f);
	}
}

void SampleApp::CheckKeyboardInput()
{
	if (glfwGetKey(m_window, GLFW_KEY_ESCAPE) == GLFW_PRESS)
	{
		glfwSetWindowShouldClose(m_window, 1);
	}
}

bool SampleApp::OnSRDEnable()
{
	SonyOzResult result = SonyOzResult::SUCCESS;

	result = LinkXrLibrary("Spatial Reality Display");
	if (result != SonyOzResult::SUCCESS) {
		std::cout << "Runtime is not found." << std::endl;
		return false;
	}

	SonyOzDeviceInfo deviceInfo[3];

	// Get the information of the connected SR Display.
	uint64_t deviceInfoSize = uint64_t(std::size(deviceInfo));
	result = sony::oz::xr_runtime::EnumerateDevices("Spatial Reality Display", deviceInfoSize, deviceInfo);

	if (result != SonyOzResult::SUCCESS || deviceInfoSize <= 0) {
		std::cout << "There are no SR Displays." << std::endl;
		return false;
	}


	m_srdRect = deviceInfo[0].target_monitor_rectangle;
	m_deviceWidth = m_srdRect.right - m_srdRect.left;
	m_deviceHeight = m_srdRect.bottom - m_srdRect.top;

	result = CreateSession("Spatial Reality Display", &deviceInfo[0], RUNTIME_OPTION_IS_XR_CONTENT, PLATFORM_OPTION_NONE, &m_sessionHandle);

	if (result != SonyOzResult::SUCCESS) {
		std::cout << "CreateSession Error." << std::endl;
		return false;
	}

	result = BeginSession(m_sessionHandle);

	if (result != SonyOzResult::SUCCESS) {
		std::cout << "BeginSession Error." << std::endl;
		return false;
	}

	if (utility::WaitUntilRunningState(m_sessionHandle) == false) {
		std::cout << "WaitUntilRunningState Error." << std::endl;
		return false;
	}

	// check if current SRD runtime supports HighImageQuality or not.
	result = SetHighImageQuality("Spatial Reality Display", m_sessionHandle, true);
	if (result == SonyOzResult::SUCCESS) {
		result = GetHighImageQuality("Spatial Reality Display", m_sessionHandle, &m_bIsHighImageQualitySupported);
		if (result != SonyOzResult::SUCCESS) {
			m_bIsHighImageQualitySupported = false;
		}
	}
	std::cout << "High Image Quality mode is " << (m_bIsHighImageQualitySupported ? "supported." : "not supported.") << std::endl;

	// check if current SRD runtime supports PerformancePriority or not.
	result = GetPerformancePriority("Spatial Reality Display", m_sessionHandle, &m_bIsPerformancePrioritySupported);
	if (result != SonyOzResult::SUCCESS) {
		m_bIsPerformancePrioritySupported = false;
	}
	std::cout << "PerformancePriority mode is " << (m_bIsPerformancePrioritySupported ? "supported." : "not supported.") << std::endl;

	// check if current SRD runtime supports Wallmount mode or not.
	if (wallmount_enabled) {
		result = SetWallmountMode("Spatial Reality Display", m_sessionHandle, true);
		if (result == SonyOzResult::SUCCESS) {
			result = GetWallmountMode("Spatial Reality Display", m_sessionHandle, &m_bIsWallmountSupported);
			if (result != SonyOzResult::SUCCESS) {
				m_bIsWallmountSupported = false;
			}
		}
	}
	std::cout << "Wallmount mode is " << (m_bIsWallmountSupported ? "supported." : "not supported.") << std::endl;

	// force HighImageQualitySupport on
	if (m_bIsPerformancePrioritySupported || m_bIsWallmountSupported) {
		m_bIsHighImageQualitySupported = true;
	}

	if (GetDisplaySpec(m_sessionHandle, &m_srdSpec) != SonyOzResult::SUCCESS)
	{
		std::cout << "GetDisplaySpec Error." << std::endl;
		return false;
	}
	m_srdScreenWidthCm = m_srdSpec.display_size.width_m * 100.0f;
	m_srdScreenHeightCm = m_srdSpec.display_size.height_m * 100.0f;
//	m_srdHeightCm = m_srdScreenHeightCm * sin(m_srdSpec.display_tilt_rad);
//	m_srdDepthCm = m_srdScreenHeightCm * cos(m_srdSpec.display_tilt_rad);
	float tilt_rad = m_bIsWallmountSupported ? PI/2 : m_srdSpec.display_tilt_rad;
	m_srdHeightCm = m_srdScreenHeightCm * sin(tilt_rad);
	m_srdDepthCm = m_srdScreenHeightCm * cos(tilt_rad);

	EnableStereo(m_sessionHandle, true);


	return true;
}

void SampleApp::OnSRDDisable()
{
	SonyOzResult result = SonyOzResult::SUCCESS;

	EnableStereo(m_sessionHandle, false);
	std::cout << "Session disconnected." << std::endl;

	// terminating process.
	result = EndSession(m_sessionHandle);

	if (result != SonyOzResult::SUCCESS) {
		std::cout << "EndSession Error." << std::endl;
	}
	result = DestroySession(&m_sessionHandle);
	if (result != SonyOzResult::SUCCESS) {
		std::cout << "DestroySession Error." << std::endl;
	}
}

void SampleApp::OnSetupSRDWindow()
{
	HWND hwnd = glfwGetWin32Window(m_window);
	SetWindowLong(hwnd, GWL_STYLE, WS_POPUP | WS_VISIBLE);
	MoveWindow(hwnd, m_srdRect.left, m_srdRect.top, m_srdRect.right - m_srdRect.left, m_srdRect.bottom - m_srdRect.top, TRUE);
}

void SampleApp::OnSetupSRDFramebuffer()
{
	uint32_t div = (m_bIsPerformancePrioritySupported) ? 2 : 1;
	m_leftTarget = m_glManager.CreateFrameBuffer(m_deviceWidth / div, m_deviceHeight / div);
	m_rightTarget = m_glManager.CreateFrameBuffer(m_deviceWidth / div, m_deviceHeight / div);
	m_leftHmTarget = m_glManager.CreateFrameBuffer(m_deviceWidth / div, m_deviceHeight / div, false);
	m_rightHmTarget = m_glManager.CreateFrameBuffer(m_deviceWidth / div, m_deviceHeight / div, false);
	m_sideBySideTarget = m_glManager.CreateFrameBuffer(m_deviceWidth * 2 / div, m_deviceHeight / div, false);
	m_compositeTarget = m_glManager.CreateFrameBuffer(m_deviceWidth, m_deviceHeight, false);
}


int main(int argc, char** argv)
{
	GLFWwindow* window = nullptr;

	if (argc == 2 && std::string(argv[1]) == "--wallmount") {
		wallmount_enabled = true;
	}

	if (!glfwInit())
	{
		return -1;
	}

	glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
	glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 3);
	glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_FALSE);
	glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE);
#if defined(_DEBUG)
	glfwWindowHint(GLFW_OPENGL_DEBUG_CONTEXT, true);
#endif

	window = glfwCreateWindow(640, 480, "SRD SDK Sample", nullptr, nullptr);
	if (!window)
	{
		glfwTerminate();
		return -1;
	}

	glfwMakeContextCurrent(window);

	if (gl3wInit() != 0)
	{
		return -1;
	}

	if (!gl3wIsSupported(3, 2))
	{
		return -1;
	}

	std::cout << "Rendering environment info:" << std::endl;
	std::cout << "- Renderer: " << glGetString(GL_RENDERER) << std::endl;
	std::cout << "- Vendor: " << glGetString(GL_VENDOR) << std::endl;
	std::cout << "- OpenGL ver.: " << glGetString(GL_VERSION) << std::endl;
	std::cout << "- GLSL ver.: " << glGetString(GL_SHADING_LANGUAGE_VERSION) << std::endl;

	{
		SampleApp app(window);
		if (!app.Initialize())
		{
			return -1;
		}

		while (!glfwWindowShouldClose(window))
		{
			app.Run();

			glfwSwapBuffers(window);
			glfwPollEvents();
		}
	}

	glfwTerminate();

	return 0;
}
