#pragma once

#include <cstdint>
#include <memory>

#include <GL/gl3w.h>
#include <glm/glm.hpp>

constexpr float PI = 3.141592f;

struct Vertex
{
	glm::vec3	m_position;
	glm::vec3	m_normal;
};

struct QuadVertex
{
	glm::vec2	m_position;
	glm::vec2	m_uv;
};

struct Quad
{
	Quad()
	{
		// Left Top
		m_quad[0].m_position = glm::vec2(-1.0f, 1.0f);
		m_quad[0].m_uv = glm::vec2(0.0f, 1.0f);

		// Left Bottom
		m_quad[1].m_position = glm::vec2(-1.0f, -1.0f);
		m_quad[1].m_uv = glm::vec2(0.0f, 0.0f);

		// Right Top
		m_quad[2].m_position = glm::vec2(1.0f, 1.0f);
		m_quad[2].m_uv = glm::vec2(1.0f, 1.0f);

		// Right Bottom
		m_quad[3].m_position = glm::vec2(1.0f, -1.0f);
		m_quad[3].m_uv = glm::vec2(1.0f, 0.0f);
	}

	/*
	* [0]: Left Top
	* [1]: Left Bottom
	* [2]: Right Top
	* [3]: Right Bottom
	*/
	QuadVertex	m_quad[4];
};

struct ShaderProgram
{
	ShaderProgram();
	~ShaderProgram();

	ShaderProgram(const ShaderProgram&) = delete;
	ShaderProgram& operator =(const ShaderProgram&) = delete;

	GLuint	m_prog = 0;
};

struct DrawObject
{
	DrawObject();
	~DrawObject();

	DrawObject(const DrawObject&) = delete;
	DrawObject& operator =(const DrawObject&) = delete;

	GLuint	m_vbo = 0;
	GLuint	m_ibo = 0;
	GLuint	m_vao = 0;

	uint32_t	m_numIndices = 0;

	glm::mat4	m_transform = glm::mat4(1);
	glm::vec4	m_diffuse = glm::vec4(1);
};


struct FrameBuffer
{
	FrameBuffer();
	~FrameBuffer();

	FrameBuffer(const FrameBuffer&) = delete;
	FrameBuffer& operator =(const FrameBuffer&) = delete;

	GLuint	m_fbo = 0;
	GLuint	m_texture = 0;
	GLuint	m_depthStencil = 0;

	uint32_t	m_width = 0;
	uint32_t	m_height = 0;
};


struct SceneInfo
{
	glm::mat4	m_view;
	glm::mat4	m_projection;

	glm::vec3	m_lightDir[4];
	glm::vec3	m_lightColor[4];
};

class GLManager
{
public:
	~GLManager();

	bool Setup();

	std::shared_ptr<ShaderProgram> CreateShaderProgram(const char* vsCode, const char* fsCode);
	std::shared_ptr<FrameBuffer> CreateFrameBuffer(uint32_t w, uint32_t h, bool enableDepth = true);

	std::shared_ptr<DrawObject> CreatePlane(float w, float h);
	std::shared_ptr<DrawObject> CreateCube(float s);
	std::shared_ptr<DrawObject> CreateSphere(float r);

	void Draw(const DrawObject& drawObject, const SceneInfo& sceneInfo);

private:

	GLuint CreateShader(GLenum shaderType, const char* code);
	bool LinkShaderProgram(GLuint prog, GLuint vs, GLuint fs);

	void SetupVAO(DrawObject& drawObject);

private:
	struct StandardMaterialShader
	{
		std::shared_ptr<ShaderProgram>	m_shader;

		GLint m_uWV;
		GLint m_uWVP;

		GLint m_uDiffuse;
		GLint m_uLightDir[4];
		GLint m_uLightColor[4];
	};

	StandardMaterialShader m_standardMaterialSahder;
};
