#include <iostream>
#include <vector>
#include <cmath>

#include "GLManager.h"
#include "StandardMaterialShader.h"

#include <glm/gtc/constants.hpp>

ShaderProgram::ShaderProgram()
{
}

ShaderProgram::~ShaderProgram()
{
	if (m_prog != 0)
	{
		glDeleteProgram(m_prog);
		m_prog = 0;
	}
}

DrawObject::DrawObject()
{
}

DrawObject::~DrawObject()
{
	if (m_vao != 0)
	{
		glDeleteVertexArrays(1, &m_vao);
		m_vao = 0;
	}
	if (m_ibo != 0)
	{
		glDeleteBuffers(1, &m_ibo);
		m_ibo = 0;
	}
	if (m_vbo != 0)
	{
		glDeleteBuffers(1, &m_vbo);
		m_vbo = 0;
	}
}

GLManager::~GLManager()
{
}

bool GLManager::Setup()
{
	m_standardMaterialSahder.m_shader = CreateShaderProgram(StandardMaterialVertexShader, StandardMaterialFragmentShader);
	if (m_standardMaterialSahder.m_shader == nullptr)
	{
		return false;
	}

	// Setup Standard Material Shader
	{
		auto prog = m_standardMaterialSahder.m_shader->m_prog;
		m_standardMaterialSahder.m_uWV = glGetUniformLocation(prog, "u_WV");
		m_standardMaterialSahder.m_uWVP = glGetUniformLocation(prog, "u_WVP");
		m_standardMaterialSahder.m_uDiffuse = glGetUniformLocation(prog, "u_Diffuse");
		m_standardMaterialSahder.m_uLightDir[0] = glGetUniformLocation(prog, "u_LightDir[0]");
		m_standardMaterialSahder.m_uLightDir[1] = glGetUniformLocation(prog, "u_LightDir[1]");
		m_standardMaterialSahder.m_uLightDir[2] = glGetUniformLocation(prog, "u_LightDir[2]");
		m_standardMaterialSahder.m_uLightDir[3] = glGetUniformLocation(prog, "u_LightDir[3]");
		m_standardMaterialSahder.m_uLightColor[0] = glGetUniformLocation(prog, "u_LightColor[0]");
		m_standardMaterialSahder.m_uLightColor[1] = glGetUniformLocation(prog, "u_LightColor[1]");
		m_standardMaterialSahder.m_uLightColor[2] = glGetUniformLocation(prog, "u_LightColor[2]");
		m_standardMaterialSahder.m_uLightColor[3] = glGetUniformLocation(prog, "u_LightColor[3]");
	}

	return true;
}

std::shared_ptr<DrawObject> GLManager::CreatePlane(float w, float h)
{
	auto drawObject = std::make_shared<DrawObject>();

	Vertex vertices[] = {
		Vertex{glm::vec3(-w / 2,  h / 2, 0), glm::vec3(0, 0, 1) },
		Vertex{glm::vec3(-w / 2, -h / 2, 0), glm::vec3(0, 0, 1) },
		Vertex{glm::vec3(w / 2, -h / 2, 0), glm::vec3(0, 0, 1) },
		Vertex{glm::vec3(w / 2,  h / 2, 0), glm::vec3(0, 0, 1) },
	};
	glGenBuffers(1, &drawObject->m_vbo);
	glBindBuffer(GL_ARRAY_BUFFER, drawObject->m_vbo);
	glBufferData(GL_ARRAY_BUFFER, std::size(vertices) * sizeof(Vertex), vertices, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	uint32_t indices[] = {
		0, 1, 2,
		2, 3, 0,
	};
	glGenBuffers(1, &drawObject->m_ibo);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, drawObject->m_ibo);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, std::size(indices) * sizeof(uint32_t), indices, GL_STATIC_DRAW);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	drawObject->m_numIndices = (uint32_t)std::size(indices);

	SetupVAO(*drawObject);

	return drawObject;
}

std::shared_ptr<DrawObject> GLManager::CreateCube(float s)
{
	auto drawObject = std::make_shared<DrawObject>();

	float hs = s / 2;

	Vertex vertices[] = {
		// Front
		Vertex{glm::vec3(-hs,  hs, hs), glm::vec3(0, 0, 1) },
		Vertex{glm::vec3(-hs, -hs, hs), glm::vec3(0, 0, 1) },
		Vertex{glm::vec3(hs, -hs, hs), glm::vec3(0, 0, 1) },
		Vertex{glm::vec3(hs,  hs, hs), glm::vec3(0, 0, 1) },
		// Right
		Vertex{glm::vec3(hs,  hs,  hs), glm::vec3(1, 0, 0) },
		Vertex{glm::vec3(hs, -hs,  hs), glm::vec3(1, 0, 0) },
		Vertex{glm::vec3(hs, -hs, -hs), glm::vec3(1, 0, 0) },
		Vertex{glm::vec3(hs,  hs, -hs), glm::vec3(1, 0, 0) },
		// Back
		Vertex{glm::vec3(hs,  hs, -hs), glm::vec3(0, 0, -1) },
		Vertex{glm::vec3(hs, -hs, -hs), glm::vec3(0, 0, -1) },
		Vertex{glm::vec3(-hs, -hs, -hs), glm::vec3(0, 0, -1) },
		Vertex{glm::vec3(-hs,  hs, -hs), glm::vec3(0, 0, -1) },
		// Left
		Vertex{glm::vec3(-hs,  hs, -hs), glm::vec3(-1, 0, 0) },
		Vertex{glm::vec3(-hs, -hs, -hs), glm::vec3(-1, 0, 0) },
		Vertex{glm::vec3(-hs, -hs,  hs), glm::vec3(-1, 0, 0) },
		Vertex{glm::vec3(-hs,  hs,  hs), glm::vec3(-1, 0, 0) },
		// Top
		Vertex{glm::vec3(-hs, hs, -hs), glm::vec3(0, 1, 0) },
		Vertex{glm::vec3(-hs, hs,  hs), glm::vec3(0, 1, 0) },
		Vertex{glm::vec3(hs, hs,  hs), glm::vec3(0, 1, 0) },
		Vertex{glm::vec3(hs, hs, -hs), glm::vec3(0, 1, 0) },
		// Bottom
		Vertex{glm::vec3(-hs, -hs,  hs), glm::vec3(0, -1, 0) },
		Vertex{glm::vec3(-hs, -hs, -hs), glm::vec3(0, -1, 0) },
		Vertex{glm::vec3(hs, -hs, -hs), glm::vec3(0, -1, 0) },
		Vertex{glm::vec3(hs, -hs,  hs), glm::vec3(0, -1, 0) },
	};
	glGenBuffers(1, &drawObject->m_vbo);
	glBindBuffer(GL_ARRAY_BUFFER, drawObject->m_vbo);
	glBufferData(GL_ARRAY_BUFFER, std::size(vertices) * sizeof(Vertex), vertices, GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	uint32_t indices[] = {
		// Front
		0, 1, 2,
		2, 3, 0,
		// Right
		4, 5, 6,
		6, 7, 4,
		// Back
		8, 9, 10,
		10, 11, 8,
		// Left
		12, 13, 14,
		14, 15, 12,
		// Top
		16, 17, 18,
		18, 19, 16,
		// Bottom
		20, 21, 22,
		22, 23, 20,
	};
	glGenBuffers(1, &drawObject->m_ibo);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, drawObject->m_ibo);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, std::size(indices) * sizeof(uint32_t), indices, GL_STATIC_DRAW);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	drawObject->m_numIndices = (uint32_t)std::size(indices);

	SetupVAO(*drawObject);

	return drawObject;
}

std::shared_ptr<DrawObject> GLManager::CreateSphere(float r)
{
	constexpr int numLatitudes = 9;
	constexpr int numLongitudes = 18;

	auto drawObject = std::make_shared<DrawObject>();

	std::vector<Vertex> vertices = {};
	for (int i = 0; i <= numLongitudes; i++)
	{
		float phi = i * 2.0f * PI / (float)numLongitudes;
		for (int j = 0; j < numLatitudes; j++)
		{
			float theta = j * PI / (float)(numLatitudes - 1);

			float x = sinf(theta) * cosf(phi);
			float y = cosf(theta);
			float z = sinf(theta) * sinf(phi);
			vertices.push_back(Vertex{ glm::vec3(x * r, y * r, z * r), glm::vec3(x, y, z) });
		}
	}
	glGenBuffers(1, &drawObject->m_vbo);
	glBindBuffer(GL_ARRAY_BUFFER, drawObject->m_vbo);
	glBufferData(GL_ARRAY_BUFFER, vertices.size() * sizeof(Vertex), vertices.data(), GL_STATIC_DRAW);
	glBindBuffer(GL_ARRAY_BUFFER, 0);

	std::vector<uint32_t> indices = {};
	for (int i = 0; i < numLongitudes; i++)
	{
		for (int j = 0; j < numLatitudes; j++)
		{
			int idx = j * numLongitudes + i;
			indices.push_back(idx);
			indices.push_back(idx + 1);
			indices.push_back(idx + numLatitudes + 1);

			indices.push_back(idx + numLatitudes + 1);
			indices.push_back(idx + numLatitudes);
			indices.push_back(idx);
		}
	}
	glGenBuffers(1, &drawObject->m_ibo);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, drawObject->m_ibo);
	glBufferData(GL_ELEMENT_ARRAY_BUFFER, indices.size() * sizeof(uint32_t), indices.data(), GL_STATIC_DRAW);
	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);

	drawObject->m_numIndices = (uint32_t)indices.size();

	SetupVAO(*drawObject);

	return drawObject;
}

void GLManager::Draw(const DrawObject& drawObject, const SceneInfo& sceneInfo)
{
	const auto vp = sceneInfo.m_projection * sceneInfo.m_view;
	const auto wv = sceneInfo.m_view * drawObject.m_transform;
	const auto wvp = vp * drawObject.m_transform;

	glUseProgram(m_standardMaterialSahder.m_shader->m_prog);
	glBindVertexArray(drawObject.m_vao);

	{
		auto prog = m_standardMaterialSahder.m_shader->m_prog;
		glUniformMatrix4fv(m_standardMaterialSahder.m_uWV, 1, GL_FALSE, &wv[0][0]);
		glUniformMatrix4fv(m_standardMaterialSahder.m_uWVP, 1, GL_FALSE, &wvp[0][0]);

		glUniform4fv(m_standardMaterialSahder.m_uDiffuse, 1, &drawObject.m_diffuse[0]);

		for (int i = 0; i < 4; i++)
		{
			glm::vec3 lightDir = glm::mat3(sceneInfo.m_view) * sceneInfo.m_lightDir[i];
			glUniform3fv(m_standardMaterialSahder.m_uLightDir[i], 1, &lightDir[0]);
			glUniform3fv(m_standardMaterialSahder.m_uLightColor[i], 1, &sceneInfo.m_lightColor[i][0]);
		}
	}

	glDrawElements(GL_TRIANGLES, drawObject.m_numIndices, GL_UNSIGNED_INT, (const void*)0);

	glBindVertexArray(0);
	glUseProgram(0);
}

void GLManager::SetupVAO(DrawObject& drawObject)
{
	GLint inPos = glGetAttribLocation(m_standardMaterialSahder.m_shader->m_prog, "in_Pos");
	GLint inNor = glGetAttribLocation(m_standardMaterialSahder.m_shader->m_prog, "in_Nor");

	glGenVertexArrays(1, &drawObject.m_vao);
	glBindVertexArray(drawObject.m_vao);

	glBindBuffer(GL_ARRAY_BUFFER, drawObject.m_vbo);

	glVertexAttribPointer(inPos, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), (const void*)0);
	glEnableVertexAttribArray(inPos);

	glVertexAttribPointer(inNor, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex), (const void*)(sizeof(glm::vec3)));
	glEnableVertexAttribArray(inNor);

	glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, drawObject.m_ibo);

	glBindVertexArray(0);
}

GLuint GLManager::CreateShader(GLenum shaderType, const char* code)
{
	GLuint shader = glCreateShader(shaderType);
	if (shader == 0)
	{
		std::cout << "Failed to create shader.\n";
		return 0;
	}
	const char* codes[] = { code };
	GLint codesLen[] = { GLint(strlen(code)) };
	glShaderSource(shader, 1, codes, codesLen);
	glCompileShader(shader);

	GLint infoLength;
	glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &infoLength);
	if (infoLength != 0)
	{
		std::vector<char> info;
		info.reserve(infoLength + 1);
		info.resize(infoLength);

		GLsizei len;
		glGetShaderInfoLog(shader, infoLength, &len, &info[0]);
		if (info[infoLength - 1] != '\0')
		{
			info.emplace_back('\0');
		}

		std::cout << &info[0] << "\n";
	}

	GLint compileStatus;
	glGetShaderiv(shader, GL_COMPILE_STATUS, &compileStatus);
	if (compileStatus != GL_TRUE)
	{
		glDeleteShader(shader);
		std::cout << "Failed to compile shader.\n";
		return 0;
	}

	return shader;
}

bool GLManager::LinkShaderProgram(GLuint prog, GLuint vs, GLuint fs)
{
	glAttachShader(prog, vs);
	glAttachShader(prog, fs);
	glLinkProgram(prog);

	GLint infoLength;
	glGetProgramiv(prog, GL_INFO_LOG_LENGTH, &infoLength);
	if (infoLength != 0)
	{
		std::vector<char> info;
		info.reserve(infoLength + 1);
		info.resize(infoLength);

		GLsizei len;
		glGetProgramInfoLog(prog, infoLength, &len, &info[0]);
		if (info[infoLength - 1] != '\0')
		{
			info.emplace_back('\0');
		}

		std::cout << &info[0] << "\n";
	}

	GLint linkStatus;
	glGetProgramiv(prog, GL_LINK_STATUS, &linkStatus);
	if (linkStatus != GL_TRUE)
	{
		glDeleteShader(vs);
		glDeleteShader(fs);
		std::cout << "Failed to link shader.\n";
		return false;
	}

	return true;
}

std::shared_ptr<ShaderProgram> GLManager::CreateShaderProgram(const char* vsCode, const char* fsCode)
{
	GLuint vs = CreateShader(GL_VERTEX_SHADER, vsCode);
	GLuint fs = CreateShader(GL_FRAGMENT_SHADER, fsCode);

	std::shared_ptr<ShaderProgram> prog;
	if (vs != 0 && fs != 0)
	{
		prog = std::make_shared<ShaderProgram>();
		prog->m_prog = glCreateProgram();
		if (!LinkShaderProgram(prog->m_prog, vs, fs))
		{
			prog = nullptr;
		}
	}

	if (vs != 0) { glDeleteShader(vs); }
	if (fs != 0) { glDeleteShader(fs); }

	return prog;
}

std::shared_ptr<FrameBuffer> GLManager::CreateFrameBuffer(uint32_t w, uint32_t h, bool enableDepth)
{
	auto framebuffer = std::make_shared<FrameBuffer>();

	// Setup color texture
	glGenTextures(1, &framebuffer->m_texture);
	glBindTexture(GL_TEXTURE_2D, framebuffer->m_texture);
	glTexImage2D(GL_TEXTURE_2D, 0, GL_RGBA, w, h, 0, GL_RGBA, GL_UNSIGNED_BYTE, nullptr);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAX_LEVEL, 0);
	glBindTexture(GL_TEXTURE_2D, 0);

	if (enableDepth)
	{
		// Setup depth stencil texture
		glGenRenderbuffers(1, &framebuffer->m_depthStencil);
		glBindRenderbuffer(GL_RENDERBUFFER, framebuffer->m_depthStencil);
		glRenderbufferStorage(GL_RENDERBUFFER, GL_DEPTH24_STENCIL8, w, h);
		glBindRenderbuffer(GL_RENDERBUFFER, 0);
	}

	// Setup framebuffer object
	glGenFramebuffers(1, &framebuffer->m_fbo);
	glBindFramebuffer(GL_FRAMEBUFFER, framebuffer->m_fbo);
	glFramebufferTexture2D(GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_TEXTURE_2D, framebuffer->m_texture, 0);
	if (enableDepth)
	{
		glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, framebuffer->m_depthStencil);
		glFramebufferRenderbuffer(GL_FRAMEBUFFER, GL_STENCIL_ATTACHMENT, GL_RENDERBUFFER, framebuffer->m_depthStencil);
	}
	glBindFramebuffer(GL_FRAMEBUFFER, 0);

	framebuffer->m_width = w;
	framebuffer->m_height = h;

	return framebuffer;
}

FrameBuffer::FrameBuffer()
{
}

FrameBuffer::~FrameBuffer()
{
	if (m_fbo != 0)
	{
		glDeleteFramebuffers(1, &m_fbo);
		m_fbo = 0;
	}

	if (m_texture != 0)
	{
		glDeleteTextures(1, &m_texture);
		m_texture = 0;
	}

	if (m_depthStencil != 0)
	{
		glDeleteRenderbuffers(1, &m_depthStencil);
		m_depthStencil = 0;
	}
}
