# Spatial Reality Display サンプルマニュアル

Spatial Reality Display サンプルについて解説します。

## Spatial Reality Display サンプルの概要

このサンプルでは OpenGL を用い Spatial Reality Display 上での描画を行います。
使用する OpenGL のバージョンは 3.3 以降となります。

### 開発環境

サンプルは以下の環境を想定しています。

- Visual Studio 2019
- Spatial Reality Display SDK 2.5.0
- Spatial Reality Display Settings Installer 2.5.0
- Native API 2.0 (XR_API_2.4.zip)

### 使用するライブラリ

以下のライブラリを使用します。

|ライブラリ名|Version|機能|URL|
|---|---|---|---|
|GLFW|3.3.9|OpenGL の初期化|<https://www.glfw.org/>|
|GLM|0.9.9.8|3Dのための数学処理|<https://github.com/g-truc/glm>|

また、OpenGL 3.3 の初期化処理として [gl3w](https://github.com/skaslev/gl3w) で生成したコードを使用します。
gl3wのhash: 3a33275633ce4be433332dc776e6a5b3bdea6506
生成日: 2024/02/16

これらのライブラリは third_pary ディレクトリに配置されます。

### ソースの構成

|ソース|機能|
|---|---|
|main.cpp|サンプル本体|
|GLManager.h/.cpp| OpenGL の簡易マネージャー |
|StandardMaterialShader.h/.cpp| 通常描画のシェーダーコード |
|QuadCopyShader.h/.cpp| 画面コピーのシェーダーコード |
|HomographyCopyShader.h/.cpp| ホモグラフィ変換のシェーダーコード |

このマニュアルではサンプル本体について説明します。

### サンプルの大まかな流れ

サンプルの大まかな流れは以下のようになります。

1. Spatial Reality Display 初期化
2. ユーザー入力確認・マウスオブジェクトの位置を更新
3. Spatial Reality Display からヘッドトラッキング情報を取得
4. ヘッドトラッキング情報から描画用の View/Projection マトリクスを作成
5. 左目用の描画
6. 右目用の描画
7. 左右目用の描画結果をホモグラフィ変換
8. ホモグラフィ変換の結果を Spatial Reality Display へ送信
9. 描画面更新

2 - 9 の処理を繰り返します。

## Spatial Reality Display の初期化、終了処理

### Spatial Reality Display の初期化

Spatial Reality Display の初期化は `SampleApp::OnSRDEnable()` で行っています。

最初に Spatial Reality Display のランタイムをロードします。

```cpp
result = LinkXrLibrary("Spatial Reality Display");
if (result != SonyOzResult::SUCCESS) {
    std::cout << "Runtime is not found." << std::endl;
    return false;
}
```

接続されている Spatial Reality Display デバイスを列挙し、使用するデバイスを決定します。
サンプルでは最初に列挙されたデバイスを使用します。

`SonyOzDeviceInfo` には Spatial Reality Display デバイスに関する情報が入っています。
配置情報(`target_monitor_rectangle`) は描画用ウィンドウを Spatial Reality Display モニターのへ配置するために使用します。

```cpp
SonyOzDeviceInfo deviceInfo[3];

uint64_t deviceInfoSize = uint64_t(std::size(deviceInfo));
result = sony::oz::xr_runtime::EnumerateDevices("Spatial Reality Display", deviceInfoSize, deviceInfo);

if (result != SonyOzResult::SUCCESS || deviceInfoSize <= 0) {
    std::cout << "There are no SR Displays." << std::endl;
    return false;
}
```

Spatial Reality Display デバイスとのセッションを開始します。

セッション開始(`BeginSession`)すると `SonyOzSessionHandle` が取得できます。

このハンドルで Spatial Reality Display の各種操作を行います。

```cpp
result = CreateSession("Spatial Reality Display", &deviceInfo[0], RUNTIME_OPTION_IS_XR_CONTENT, PLATFORM_OPTION_NONE, &m_sessionHandle);

if (result != SonyOzResult::SUCCESS) {
    std::cout << "CreateSession Error." << std::endl;
    return false;
}

result = BeginSession(m_sessionHandle);

if (result != SonyOzResult::SUCCESS) {
    std::cout << "BeginSession Error." << std::endl;
    return false;
}

if (utility::WaitUntilRunningState(m_sessionHandle) == false) {
    std::cout << "WaitUntilRunningState Error." << std::endl;
    return false;
}
```

Spatial Reality Display のスペック(`SonyOzDisplaySpec`)を取得します。

スペックには Spatial Reality Display の寸法(`display_size`)、画面の傾き(`display_tilt_rad`)を取得できます。
SR-1 と SR-2 では寸法が違うので、ここで取得した値を参照しシーンを構築します(表示するキューブの大きさを実寸に合わせ調整します)。

寸法と画面の傾きは、3D空間の描画、ホモグラフィ変換に使用します。

寸法はメートル単位で取得できますが、このサンプルではセンチメートル単位で 3D 空間を構成するため100倍して使用します。

```cpp
if (GetDisplaySpec(m_sessionHandle, &m_srdSpec) != SonyOzResult::SUCCESS)
{
    std::cout << "GetDisplaySpec Error." << std::endl;
    return false;
}
```

Spatial Reality Display で立体視を開始します。

```cpp
EnableStereo(m_sessionHandle, true);
```

### Spatial Reality Display の終了処理

Spatial Reality Display の終了処理は `SampleApp::OnSRDDisable()` で行っています。

以下のことを行います。

1. 立体視を終了 (`EnableStereo`)
2. セッションを終了 (`EndSession`)
3. セッション用ハンドルを破棄 (`DestroySession`)

```cpp
EnableStereo(m_sessionHandle, false);
std::cout << "Session disconnected." << std::endl;

// terminating process.
result = EndSession(m_sessionHandle);

if (result != SonyOzResult::SUCCESS) {
    std::cout << "EndSession Error." << std::endl;
}
result = DestroySession(&m_sessionHandle);
if (result != SonyOzResult::SUCCESS) {
    std::cout << "DestroySession Error." << std::endl;
}
```

## 3D 描画物の準備

サンプルで表示する 3D シーンは `SampleApp::SetupScene()` で用意されます。

シーンは以下の構成となります。

- 下側の平面
- 奥側の平面
- サンプルオブジェクト（キューブ）
- マウスポインタオブジェクト

![サンプルシーン](images/img_001.png)

サンプルでは 3D シーンは使用する Spatial Reality Display の寸法に合わせて作成しています。

シーンの座標は Spatial Reality Display の手前中央を (0, 0, 0) とした右手座標系になります。

![シーンの座標系](images/img_002.png)

## マウスポインタオブジェクト

3D シーンに表示するマウスポインタ用の 3D オブジェクトの位置は `SampleApp::CheckMouseInput()` で更新しています。\
スクリーン座標でのマウスの位置を取得し、Spatial Reality Display のパネル面で表示されるように3D シーン座標に変換します。\
また、マウス入力が正しく記録されているかを確認するために、左クリック押下時にポインタオブジェクトの色を変更しています。

```cpp
void SampleApp::CheckMouseInput()
{
    // Get mouse position in screen coordinates
    double xpos, ypos;
    glfwGetCursorPos(m_window, &xpos, &ypos);

    double xPercent = xpos / (m_deviceWidth - 1);
    double yPercent = ypos / (m_deviceHeight - 1);

    // Convert to position on the SRD plane in world coordinates
    double worldX = xPercent * m_srdScreenWidthCm - m_srdScreenWidthCm / 2;
    double worldY = m_srdHeightCm - yPercent * m_srdHeightCm;
    double worldZ = yPercent * m_srdDepthCm - m_srdDepthCm;

    m_mousePointerObj->m_transform = glm::translate(glm::mat4(1), glm::vec3(worldX, worldY, worldZ));

    // Change object color when left button is being pressed
    if (glfwGetMouseButton(m_window, GLFW_MOUSE_BUTTON_LEFT) == GLFW_PRESS)
    {
        m_mousePointerObj->m_diffuse = glm::vec4(170.0f / 255.0f, 85.0f / 255.0f, 85.0f / 255.0f, 1.0f);
    }
    else if (glfwGetMouseButton(m_window, GLFW_MOUSE_BUTTON_LEFT) == GLFW_RELEASE)
    {
        m_mousePointerObj->m_diffuse = glm::vec4(170.0f / 255.0f, 170.0f / 255.0f, 170.0f / 255.0f, 1.0f);
    }
}
```

## キーボード入力

`SampleApp::CheckKeyboardInput()` でキー入力をチェックしています。\

```cpp
void SampleApp::CheckKeyboardInput()
{
    if (glfwGetKey(m_window, GLFW_KEY_ESCAPE) == GLFW_PRESS)
    {
        glfwSetWindowShouldClose(m_window, 1);
    }
}
```

- ESC キー押下時にプログラムを終了します。

## ヘッドトラッキング

3D シーンの描画に用いる View / Projection マトリクスは Spatial Reality Display によりトラッキングされた目の情報をもとに作成します。

### View マトリクスの作成

Spatial Reality Display から目の姿勢情報(`SonyOzPosef`)を取得し View マトリクスを作成します。

姿勢情報を取得するにはまず、トラッキングデータのキャッシュを更新します(`UpdateTrackingResultCache`)。

次に、更新した左右の目の情報を取得します(`GetCachedPose`)。

```cpp
UpdateTrackingResultCache(m_sessionHandle);
GetCachedPose(m_sessionHandle, SonyOzPoseId::LEFT_EYE, &left_pose, &leftValid);
GetCachedPose(m_sessionHandle, SonyOzPoseId::RIGHT_EYE, &right_pose, &rightValid);
```

姿勢情報には位置(`position`)と方向(`orientation`)があります。
座標系はシーンの座標系と反転しており、以下の図のようになっています。

![カメラの座標系](images/img_003.png)

View マトリクスの作成は `MakeViewMatrix()` で行います。

```cpp
static glm::mat4 MakeViewMatrix(const SonyOzPosef& pose)
{
    /*
     * SRD SDK 2.0 の顔の位置は、右がマイナス方向、手前がマイナスとなる
     * Maya 標準の右手座標系に変換する
     */
    auto inv_xz = glm::mat3(
        -1, 0, 0,
        0, 1, 0,
        0, 0, -1
    );

    auto camRot = glm::quat(pose.orientation.w, pose.orientation.x, pose.orientation.y, pose.orientation.z);
    auto camPos = glm::vec3(pose.position.x, pose.position.y, pose.position.z);
    camPos = (glm::mat4(inv_xz) * glm::translate(glm::mat4(1), camPos) * glm::mat4(inv_xz))[3];
    camRot = glm::quat_cast(inv_xz * glm::mat3_cast(camRot) * inv_xz);

    // SRD SDK 2.0 の単位はメートル(1.0 はセンチメートル)
    // SRD SDK 1.0 と同じ条件に合わせるためセンチメートルに変換
    camPos = camPos * 100.0f;
    auto view = glm::inverse(glm::translate(glm::mat4(1), camPos) * glm::mat4_cast(camRot));

    return view;
}
```

### Projection マトリクスの作成

透視投影変換に使用する情報(`SonyOzProjection`)を取得し、Projection マトリクスを作成します。

`SonyOzProjection` は `GetProjection` で取得できます。

```cpp
GetProjection(m_sessionHandle, SonyOzPoseId::LEFT_EYE, &left_eye_projection);
GetProjection(m_sessionHandle, SonyOzPoseId::RIGHT_EYE, &right_eye_projection);
```

Projection マトリクスの作成は `MakeProjectionMatrix` で行います。

```cpp
static glm::mat4 MakeProjectionMatrix(const SonyOzProjection& proj, float nearClip, float farClip)
{
    const float left = nearClip * tanf(proj.half_angles_left);
    const float right = nearClip * tanf(proj.half_angles_right);
    const float top = nearClip * tanf(proj.half_angles_top);
    const float bottom = nearClip * tanf(proj.half_angles_bottom);
    return glm::frustumRH(left, right, bottom, top, nearClip, farClip);
}
```

## Spatial Reality Display への描画

シーンの描画は、左右の目から見たものをそれぞれ描画します。
また、シーンの描画結果はホモグラフィ変換を行い、サイドバイサイドの結果を Spatial Reality Display Runtime へ渡します。

それぞれの描画先(レンダーターゲット)を用意します(`SampleApp::OnSetupSRDFramebuffer`)。

```cpp
m_leftTarget = m_glManager.CreateFrameBuffer(m_deviceWidth, m_deviceHeight);
m_rightTarget = m_glManager.CreateFrameBuffer(m_deviceWidth, m_deviceHeight);
m_leftHmTarget = m_glManager.CreateFrameBuffer(m_deviceWidth, m_deviceHeight, false);
m_rightHmTarget = m_glManager.CreateFrameBuffer(m_deviceWidth, m_deviceHeight, false);
m_sideBySideTarget = m_glManager.CreateFrameBuffer(m_deviceWidth * 2, m_deviceHeight, false);
m_compositeTarget = m_glManager.CreateFrameBuffer(m_deviceWidth, m_deviceHeight, false);
```

### 左右の目から見えるものの描画

シーンの描画は左右2回行います。

```cpp
// Render left view
m_sceneInfo.m_view = viewL;
m_sceneInfo.m_projection = projL;
glBindFramebuffer(GL_FRAMEBUFFER, m_leftTarget->m_fbo);
RenderScene();

// Render right view
m_sceneInfo.m_view = viewR;
m_sceneInfo.m_projection = projR;
glBindFramebuffer(GL_FRAMEBUFFER, m_rightTarget->m_fbo);
RenderScene();
```

### ホモグラフィ変換

実物の Spatial Reality Display の画面は角度がついており、そのまま描画結果を用いると歪んでしまいます。

![レンダリング結果と実際の見た目](images/img_004.png)

実物で見た際の歪みを考慮して、描画結果をホモグラフィ変換で歪ませる必要があります。

![ホモグラフィ変換](images/img_005.png)

ホモグラフィ変換に用いるマトリクスは `MakeHomographyMatrix` で作成します。
この関数は、(0, 0)-(1, 1) で構成される4頂点を、任意の4頂点へと変形させるホモグラフィマトリクスを作成します。

```cpp
static glm::mat3 MakeHomographyMatrix(const glm::vec2 viewportPoints[4])
{
    const auto p00 = viewportPoints[0];
    const auto p01 = viewportPoints[1];
    const auto p10 = viewportPoints[2];
    const auto p11 = viewportPoints[3];

    const auto x00 = p00.x;
    const auto y00 = p00.y;
    const auto x01 = p01.x;
    const auto y01 = p01.y;
    const auto x10 = p10.x;
    const auto y10 = p10.y;
    const auto x11 = p11.x;
    const auto y11 = p11.y;

    const auto a = x10 - x11;
    const auto b = x01 - x11;
    const auto c = x00 - x01 - x10 + x11;
    const auto d = y10 - y11;
    const auto e = y01 - y11;
    const auto f = y00 - y01 - y10 + y11;

    const auto h13 = x00;
    const auto h23 = y00;
    const auto h32 = (c * d - a * f) / (b * d - a * e);
    const auto h31 = (c * e - b * f) / (a * e - b * d);
    const auto h11 = x10 - x00 + h31 * x10;
    const auto h12 = x01 - x00 + h32 * x01;
    const auto h21 = y10 - y00 + h31 * y10;
    const auto h22 = y01 - y00 + h32 * y01;

    return glm::mat3(h11, h12, h13, h21, h22, h23, h31, h32, 1.0f);
}
```

最初に、Spatial Reality Display のディスプレイ面の4頂点が、レンダリングした 2D スクリーンのどの位置に表示されるか計算します。

3D シーン内の座標が、2Dスクリーンないのどこにあたるかを計算するために `WorldToViewport` を使用します。
この関数では、左上(0,0)、右下(1, 1)とした座標を計算します。

```cpp
static glm::vec2 WorldToViewport(const glm::mat4& vp, const glm::vec3& pos)
{
    glm::vec4 screenPos = vp * glm::vec4(pos, 1);
    screenPos /= screenPos.w;
    return glm::vec2((screenPos.x + 1.0f) * 0.5f, 1.0f - (screenPos.y + 1.0f) * 0.5f);
}
```

ディスプレイ面の4頂点は Spatial Reality Display の実寸を使用します。

```cpp
glm::vec3 srdDispalyPlanePositions[4];
{
    const float x0 = -m_srdScreenWidthCm * 0.5f; // Left
    const float x1 = +m_srdScreenWidthCm * 0.5f; // Right
    const float y0 = m_srdHeightCm;              // Top
    const float y1 = 0.0f;                       // Bottom
    const float z0 = 0.0f;                       // Front
    const float z1 = -m_srdDepthCm;              // Back
    srdDispalyPlanePositions[0] = glm::vec3(x0, y0, z1); // Left Top
    srdDispalyPlanePositions[1] = glm::vec3(x0, y1, z0); // Left Bottom
    srdDispalyPlanePositions[2] = glm::vec3(x1, y0, z1); // Right Top
    srdDispalyPlanePositions[3] = glm::vec3(x1, y1, z0); // Right Bottom
}

// Left eye
const auto vpLeft = projL * viewL;
const glm::vec2 viewportLeft[] =
{
    WorldToViewport(vpLeft, srdDispalyPlanePositions[0]),
    WorldToViewport(vpLeft, srdDispalyPlanePositions[1]),
    WorldToViewport(vpLeft, srdDispalyPlanePositions[2]),
    WorldToViewport(vpLeft, srdDispalyPlanePositions[3]),
};
```

次にホモグラフィマトリクスと、その逆マトリクスを計算します。

```cpp
glm::mat3 homographyL = MakeHomographyMatrix(viewportLeft);
glm::mat3 inverseHomographyL = MakeInverseHomographyMatrix(homographyL);
```

ホモグラフィ変換用のシェーダーでは、頂点シェーダーでホモグラフィ変換を、フラグメントシェーダーで逆マトリクスを用い元画像のピクセルを参照します。

今回は元画像を歪ませ、実際に見た際に歪みをなくす処理が必要となります。
これは、シェーダーにホモグラフィマトリクスを渡す際に、逆マトリクスを頂点シェーダーに渡すようにします。

```cpp
CopyHomography(*m_leftHmTarget, m_leftTarget->m_texture, hmLQuad, inverseHomographyL, homographyL);
```

### Spatial Reality Display 上での表示

ホモグラフィ変換の結果をサイドバイサイドとして配置したテクスチャを用意します。

```cpp
Quad copyLeftQuad;
copyLeftQuad.m_quad[0].m_position = glm::vec2(-1, 1);
copyLeftQuad.m_quad[1].m_position = glm::vec2(-1, -1);
copyLeftQuad.m_quad[2].m_position = glm::vec2(0, 1);
copyLeftQuad.m_quad[3].m_position = glm::vec2(0, -1);
CopyQuad(*m_sideBySideTarget, m_leftHmTarget->m_texture, copyLeftQuad);

Quad copyRightQuad;
copyRightQuad.m_quad[0].m_position = glm::vec2(0, 1);
copyRightQuad.m_quad[1].m_position = glm::vec2(0, -1);
copyRightQuad.m_quad[0].m_position = glm::vec2(1, 1);
copyRightQuad.m_quad[1].m_position = glm::vec2(1, -1);
CopyQuad(*m_sideBySideTarget, m_rightHmTarget->m_texture, copyRightQuad);
```

作成したサイドバイサイドのテクスチャを Spatial Reality Display Runtime へ送信し、合成結果を取得します。

```cpp
SubmitOpengl(m_sessionHandle, m_sideBySideTarget->m_texture, false, m_compositeTarget->m_texture);
```

合成結果を OpenGL の画面 (FBO = 0) へ描画します。

```cpp
CopyQuad(0, m_deviceWidth, m_deviceHeight, m_compositeTarget->m_texture, copyCompositeQuad);
```
