/**
 * Sony CONFIDENTIAL
 *
 * Copyright 2022-2024 Sony Corporation
 *
 * DO NOT COPY AND/OR REDISTRIBUTE WITHOUT PERMISSION.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#pragma once

#include "xr-runtime-common/xr_api_defs.h"

XR_DEFINE_HANDLE(SonyOzSessionHandle);

namespace sony::oz::xr_runtime {
  /**
   * @brief Show/Hide camera window.
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[out] enable true: show camera window, false: hide camera window.
   * @return SonyOzResult
   * @details
   * Show/Hide camera window.
   * @attention
   */
  SonyOzResult SetCameraWindowEnabled(SonyOzPlatformId platform_id, SonyOzSessionHandle session, const bool enable);

  /**
   * @brief Get head pose status.
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[in enabled true: head pose tracking is stopped, false: head pose is tracking.
   * @return SonyOzResult
   * @details
   * Get head pose status.
   * @attention
   */
  SonyOzResult GetPauseHeadPose(SonyOzPlatformId platform_id, SonyOzSessionHandle session, bool* enabled);

  /**
   * @brief Enable/Disable head pose tracking..
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[out] enabled true: Stop head pose tracking, false: Start head pose tracking.
   * @return SonyOzResult
   * @details
   * Enable/Disable head pose tracking.
   * @attention
   */
  SonyOzResult SetPauseHeadPose(SonyOzPlatformId platform_id, SonyOzSessionHandle session, const bool enabled);

  /**
   * @brief Get HighImageQuality mode.
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[in] enabled true: HighImageQuality mode is enabled, false: HighImageQuality mode is disabled or not supported.
   * @return SonyOzResult
   * @details
   * Get HighImageQuality mode.
   * @attention
   */
  SonyOzResult GetHighImageQuality(SonyOzPlatformId platform_id, SonyOzSessionHandle session, bool* enabled);

  /**
   * @brief Set HighImageQuality mode.
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[out] enabled true: HighImageQuality mode is enabled, false: HighImageQuality mode is disabled or not supported.
   * @return SonyOzResult
   * @details
   * Get HighImageQuality mode.
   * @attention
   */
  SonyOzResult SetHighImageQuality(SonyOzPlatformId platform_id, SonyOzSessionHandle session, const bool enabled);

  /**
   * @brief Get Camera Image size
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[out] width Width of a captured camera image in pixel.
   * @param[out] height Height of a captured camera image in pxel.
   * @param[out] depth Bytes per pixel.
   * @return SonyOzResult
   * @details
   * Get Camera Image size.
   * Since the size of the camera image is fixed, it is necessary to call only once first to get the size to prepare the buffer.
   * @attention
   */
  SonyOzResult GetCameraImageSize(SonyOzPlatformId platform_id, SonyOzSessionHandle session, int *width, int *height, int *depth);

  /**
   * @brief Get Camera Image.
   * @param[out] platform_id Indicating XR platform. It will always be "Spatial Reality Display"
   * @param[out] session Pass the session handle.
   * @param[in, out] buffer_size Size of prepared buffer. If the call succeeds, the actual size copied is returned.
   * @param[out] buffer Address of the allocated buffer. The memory area must be larger than the size specified by buffer_size.
   * @return SonyOzResult
   * @details
   * Get Camera Image.
   * Camera Image is copied to this buffer.
   * @attention
   */
  SonyOzResult GetCameraImageBuffer(SonyOzPlatformId platform_id, SonyOzSessionHandle session, size_t *buffer_size, void *buffer);
}
