﻿/**
 * @file xr_api_wrapper.h
 * @copyright
 *
 * Sony CONFIDENTIAL
 *
 * Copyright 2019,2020,2021 Sony Corporation
 *
 * DO NOT COPY AND/OR REDISTRIBUTE WITHOUT PERMISSION.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
#ifndef XR_API_WRAPPER_H_
#define XR_API_WRAPPER_H_

#include "log-settings/log_defs.h"
#include "xr-runtime-common/xr_api_defs.h"
#include "xr-runtime-common/xr_platform_option.h"
#include "xr-runtime-common/xr_runtime_option.h"

// = Standard Library =============================
#include <vector>

XR_DEFINE_HANDLE(SonyOzSessionHandle);

/**
 * @namespace sony::oz
 * @brief API
 */
namespace sony::oz::xr_runtime {

/******************************************************************************
 * Load/Unload xr runtime library process
 */

/**
 * \~Japanese
 * @brief ライブラリをリンクし、XR API を使用可能にする。
 * @param[in] platform_id 利用するプラットフォームのIDを指定する。
 * @return SonyOzResult
 * \~English
 * @brief Link the library and make the API available.
 * @param[in] platform_id Specify the platform ID to be used.
 * @return SonyOzResult
 */
SonyOzResult LinkXrLibrary(SonyOzPlatformId platform_id);


/**
 * \~Japanese
 * @brief XR API の使用を終了し、ライブラリをアンリンクする。
 * \~English
 * @brief Terminate use of XR API and unlink the library.
 */
void UnlinkXrLibrary();

/******************************************************************************
 * Device library process
 */

/**
 * \~Japanese
 * @brief 利用可能なデバイスの数を取得する。
 * @param[in] platform_id 利用するプラットフォームのIDを指定する。
 * @param[out] num デバイスの数が返される
 * @return SonyOzResult
 * @details
 * @attention
 * 返される値は内部で使用される dummy のデバイスを含むため、常に接続されているデバイス + 1 となります。
 * そのため、1台も接続されていないも 1 が返ります。
 * SRD が接続されてるかどうかを判断するためには、返される値が2以上であるかで判断する必要があります。
 * \~English
 * @brief Get the number of available devices.
 * @param[in] platform_id Specify the platform ID to be used.
 * @param[out] num The number of devices is returned.
 * @return SonyOzResult
 * @details
 * @attention
 * The value returned is always the connected devices + 1, since it includes the dummy devices used internally.
 * Therefore, 1 is returned even if none of the devices are connected.
 * To determine if an SRD is connected, it is necessary to determine if the returned value is greater than 1. */
SonyOzResult GetDeviceNum(SonyOzPlatformId platform_id, uint64_t* num);

/**
 * \~Japanese
 * @brief 利用可能なデバイス（ディスプレイ）の情報を取得する。
 * @param[in] platform_id 利用するプラットフォームのIDを指定する。
 * @param[in] size リストのサイズを指定する
 * @param[out] device_list デバイス（ディスプレイ）の情報（リスト）が返される
 * @return SonyOzResult
 * @details
 * 取得したデバイスリストから利用するデバイス情報をCreateSessionへ渡すこと。
 * @attention
 * 取得したデバイス情報の配列の最後の要素は内部で使用されるものです。接続されているSRDの情報は 0 ～ size ｰ 2 の範囲になります。
 * \~English
 * @brief Obtain information on available devices.
 * @param[in] platform_id Specify the platform ID to be used.
 * @param[in] size Specify array size.
 * @param[out] device_list Device information is set in the array.
 * @return SonyOzResult
 * @details
 * Passing the device information to be used from the acquired device list to CreateSession.
 * @attention
 * The last element of the array of device information retrieved is the one used internally.
 * The information on connected SRDs ranges from 0 to size-2.
 */
SonyOzResult EnumerateDevices(SonyOzPlatformId platform_id, uint64_t size,
                              SonyOzDeviceInfo* device_list);

/******************************************************************************
 * Log process
 */

/**
 * \~Japanese
 * @brief ログの出力先を指定した関数に変更する。
 * @param[in] platform_id 利用するプラットフォームのIDを指定する。
 * @param[in] callback ログの出力先に設定する関数を渡す。
 * @return SonyOzResult
 * @details
 * ログの出力先をコールバックとして登録し、本ライブラリのログを任意の場所へ出力する。
 * @attention
 * \~English
 * @brief Change the log output destination to the specified function.
 * @param[in] platform_id Specify the platform ID to be used.
 * @param[in] callback Pass the function to be set as the log output destination.
 * @return SonyOzResult
 * @details
 * Registers the log output destination as a callback and outputs the log of this library to an arbitrary location.
 * @attention
 */
SonyOzResult SetDebugLogCallback(SonyOzPlatformId platform_id,
                                 SonyOzLogSettings_LogCallback callback);

/******************************************************************************
 * Session process
 */

/**
 * \~japanese
 * @brief XR Platformとのセッションを作成する。
 * @param[in] platform_id 利用するプラットフォームのIDを指定する。
 * @param[in] device 利用するデバイスの情報を渡す。
 * @param[in] runtime_option_bit_flag
 * 利用するアプリケーションの設定フラグを渡す。
 * 通常は RUNTIME_OPTION_IS_XR_CONTENT を指定する。
 * @param[in] platform_option_bit_flag
 * 利用するプラットフォームの設定フラグを渡す。
 * 通常は PLATFORM_OPTION_NONE を指定する。
 * @param[out] session セッションのハンドルが返される。
 * @return SonyOzResult
 * @details
 * セッションの作成には利用するデバイスの情報が必要なため、先にEnumerateDevices（device
 * enumeration library）を実行し、デバイス情報を取得しておくこと。
 *
 * この関数は複数回実行されても動作に問題ないことが保証されている。
 *
 * 同じデバイス情報を渡した場合は同一のセッションを取得でき、
 * 別のデバイス情報を渡した場合は別のセッションが作成される。
 * @attention
 * 異なるXR Platformへの複数セッションをサポートしていない。\n
 * 複数回実行した場合は、取得した全てのセッションハンドルに対してDestroySessionを実行すること。
 * \~english
 * @brief Create session with XR Platform.
 * @param[in] platform_id Specify the ID of the platform to be used.
 * @param[in] device Pass the information of the device to be used.
 * @param[in] runtime_option_bit_flag Pass the settings flag of runtime (usually RUNTIME_OPTION_IS_XR_CONTENT is specified).
 * @param[in] platform_option_bit_flag Pass the settings flag of platform (usually PLATFORM_OPTION_NONE is specified).
 * @param[out] session  A session handle is returned.
 * @return SonyOzResult
 * @details
 * Since information on the device to be used is required to create a session, EnumerateDevices (device enumeration library) to obtain device information.
 *
 * This function is guaranteed to work even if it is executed multiple times.
 *
 * If the same device information is passed, the same session is obtained; if different device information is passed, a different session is created.
 * @attention
 * Multiple sessions to different XR Platforms are not supported. If multiple sessions are executed, DestroySession should be executed for all session handles obtained.
 */
SonyOzResult CreateSession(SonyOzPlatformId platform_id,
                           const SonyOzDeviceInfo* device,
                           RUNTIME_OPTION_BIT_FLAG runtime_option_bit_flag,
                           PLATFORM_OPTION_BIT_FLAG platform_option_bit_flag,
                           SonyOzSessionHandle* session);

/**
 * \~japanese
 * @brief XR Platformとのセッションを破棄する。
 * @param[in] session セッションのハンドルを渡す。
 * @return SonyOzResult
 * @details
 * この関数は複数回実行されても動作に問題ないことが保証されている。
 * @attention
 * CreateSessionを利用して複数のセッションハンドルを取得した場合は、
 * 取得した全てのセッションハンドルに対して実行すること。
 * \~English
 * @brief Destroy session with XR Platform.
 * @param[in] session Pass session handle.
 * @return SonyOzResult
 * @details
 * This function is guaranteed to work even if it is executed multiple times.
 * @attention.
 * If multiple session handles are obtained using CreateSession, then
 * execute this function for all acquired session handles.
 */
SonyOzResult DestroySession(SonyOzSessionHandle* session);

/**
 * \~Japanese
 * @brief XR Platformとのセッションを開始する。
 * @param[in] session セッションのハンドルを渡す。
 * @return SonyOzResult
 * @details
 * セッションの状態が READY の時にセッションの開始処理が実行され、\n
 * GetSessionStateにて処理の状態を確認出来る。\n
 * （セッションの状態が RUNNING である時にセッションが開始したと判断出来る。）
 *
 * この関数が実行された時からChrome Tracing Profileの出力が行われる。
 *
 * この関数は複数回実行されても動作に問題ないことが保証されている。
 * @attention
 * 複数回実行した場合は、その回数だけEndSessionを実行すること。
 * \~English
 * @brief Start a session with the XR Platform.
 * @param[in] session Pass session handle.
 * @return SonyOzResult
 * @details
 * The session start process is executed when the session state is RUNNING, and the status of the process can be checked with GetSessionState.(When the session state is RUNNING, it can be determined that the session has started.)
 *
 * Chrome Tracing Profile is output from the time this function is executed.
 *
 * This function is guaranteed to work even if it is executed multiple times.
 * @attention.
 * If this function is executed multiple times, EndSession should be executed that number of times.
 */
SonyOzResult BeginSession(SonyOzSessionHandle session);

/**
 * \~Japanese
 * @brief XR Platformとのセッションを終了する。
 * @param[in] session セッションのハンドルを渡す。
 * @return SonyOzResult
 * @details
 * セッションの状態が RUNNING
 * の時にセッションの終了処理が実行され、(RUNNINGじゃなくても処理すべきかも)\n
 * GetSessionStateにて処理の状態を確認出来る。\n
 * （セッションの状態が READY である時にセッションが終了したと判断出来る。）
 *
 * この関数は複数回実行されても動作に問題ないことが保証されている。
 * @attention
 * BeginSessionを実行した回数だけ実行すること。\n
 * BeginSessionの実行数に満たない場合はセッションの状態が RUNNING のままとなる。
 * \~English
 * @brief Terminate the session with XR Platform.
 * @param[in] session Pass session handle.
 * @return SonyOzResult
 * @details
 * When the session state is RUNNING, the session termination process is executed, and the status of the process can be checked with GetSessionState (even if the session is not RUNNING, it should be processed). (When the session state is READY, it can be determined that the session has terminated.)
 *
 * If this function is executed multiple times, EndSession should be executed that number of times.
 * @attention
 * To be executed as many times as BeginSession is executed.\n
 * If the number of executions of BeginSession is not reached, the session status remains RUNNING.
 */
SonyOzResult EndSession(SonyOzSessionHandle session);

/**
 * \~Japanese
 * @brief セッションの状態を取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[out] session_state セッションの状態が返される。
 * @return SonyOzResult
 * @details
 * @attention
 * \~English
 * @brief Obtain session status.
 * @param[in] session Pass the session handle.
 * @param[out] session_state Session status is returned.
 * @return SonyOzResult
 * @details
 * @attention
 * \~English
  */
SonyOzResult GetSessionState(SonyOzSessionHandle session,
                             SonyOzSessionState* session_state);

/******************************************************************************
 * Tracking process
 */

/**
 * \~Japanese
 * @brief トラッキングデータのキャッシュを更新する。(Game Thread 用)
 * @param[in] session セッションのハンドルを渡す。
 * @return SonyOzResult
 * @details
 * @attention
 * セッションの状態が RUNNING の時のみ正しい結果が返ってくる。
 * それ以外の状態ではエラーが返ってくる。
 * \~English
 * @brief Update the cache of tracking data.(for Game Thread)
 * @param[in] session Pass the session handle.
 * @return SonyOzResult
 * @details
 * @attention
 * Correct results are returned only when the session state is RUNNING.
 * Otherwise, an error is returned.
 */
SonyOzResult UpdateTrackingResultCache(SonyOzSessionHandle session);

/**
 * \~Japanese
 * @brief カメラの情報を取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] pose_id 取得するカメラを指定する。
 * @param[out] pose カメラの情報が返される。
 * @param[out] is_valid カメラの情報の有効性を返す。
 * @return SonyOzResult
 * @details
 * 指定されたカメラの視点の姿勢といった情報を取得出来る。
 * @attention
 * セッションの状態が RUNNING の時のみ正しい情報を取得出来る。
 * それ以外の状態ではエラーが返ってくる。エラーの時に取得される pose
 * は不定値である。
 * \~English
 * @brief Obtain information about the camera.
 * @param[in] session Pass the session handle.
 * @param[in] pose_id Specify the camera to be obtained.
 * @param[out] pose Camera information is returned.
 * @param[out] is_valid Camera information is returned.
 * @return SonyOzResult
 * @details
 * Information such as the viewpoint posture of the specified camera can be acquired.
 * @attention
 * Correct results are returned only when the session state is RUNNING.
 * Otherwise, an error is returned.The pose obtained in case of an error is an undefined value.
 */
SonyOzResult GetCachedPose(SonyOzSessionHandle session, SonyOzPoseId pose_id,
                           SonyOzPosef* pose, bool* is_valid);

/**
 * \~Japanese
 * @brief Projectionを取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] pose_id 取得するカメラを指定する。
 * @param[out] projection Projectionが返される。
 * @return SonyOzResult
 * @details
 * 左目と右目、そしてそれぞれの中央地点のProjectionを取得する。
 * @attention
 * セッションの状態が RUNNING の時のみ正しい情報を取得出来る。
 * それ以外の状態ではエラーが返ってくる。エラーの時に取得される
 * projection_matrix は不定値である。
 * \~English
 * @brief Obtain Projection.
 * @param[in] session Pass the session handle.
 * @param[in] pose_id Specify the camera to be obtained.
 * @param[out] projection Projection is returned.
 * @return SonyOzResult
 * @details
 * Obtain the Projection of the left eye, the right eye, and the respective center point.
 * @attention
 * Correct information can be obtained only when the session state is RUNNING.
 * Otherwise, an error is returned. In case of an error, the following information is retrieved projection_matrix is undefined.
 */
SonyOzResult GetProjection(SonyOzSessionHandle session, SonyOzPoseId pose_id,
                           SonyOzProjection* projection);

/******************************************************************************
 * Rendering process
 */

/**
 * \~Japanese
 * @brief D3D11のレンダリングテクスチャとシェーダの生成を行う。（シングルパス）
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] device
 * @param[in] side_by_side 両目テクスチャのポインタを渡す。
 * @param[in] flip_y_position VertexShaderのY軸を反転させるかを渡す。
 * @param[out] target Render Targetのポインタを渡す。
 * @return SonyOzResult
 * @details
 * オフスクリーン レンダリングとシェーダー用のテクスチャを生成する。
 * @attention
 * \~English
 * @brief Generate render textures and shaders for D3D11. (single pass)
 * @param[in] session Pass the session handle.
 * @param[in] device
 * @param[in] side_by_side Pass a pointer to the SideBySide texture.
 * @param[in] flip_y_position Pass whether to invert the Y axis of the VertexShader.
 * @param[out] target Pass a pointer to the Render Target.
 * @return SonyOzResult
 * @details
 * Generate textures for off-screen rendering and shaders.
 * @attention
 */
SonyOzResult SubmitD3d11(SonyOzSessionHandle session, ID3D11Device* device,
                         ID3D11Texture2D* side_by_side, bool flip_y_position,
                         ID3D11Texture2D* target);

/**
 * \~Japanese
 * @brief D3D12のレンダリングテクスチャとシェーダの生成を行う。（シングルパス）
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] command_queue
 * @param[in] node_mask
 * @param[in] side_by_side 両目テクスチャのポインタを渡す。
 * @param[in] flip_y_position VertexShaderのY軸を反転させるかを渡す。
 * @param[in] format
 * @param[out] target Render Targetのポインタを渡す。
 * @return SonyOzResult
 * @details
 * オフスクリーン レンダリングとシェーダー用のテクスチャを生成する。
 * @attention
 * \~English
 * @brief Generate render textures and shaders for D3D11. (single pass)
 * @param[in] session Pass the session handle.
 * @param[in] command_queue
 * @param[in] node_mask
 * @param[in] side_by_side Pass a pointer to the SideBySide texture.
 * @param[in] flip_y_position Pass whether to invert the Y axis of the VertexShader.
 * @param[out] target Pass a pointer to the Render Target.
 * @return SonyOzResult
 * @details
 * Generate textures for off-screen rendering and shaders.
 * @attention
 */
SonyOzResult SubmitD3d12(SonyOzSessionHandle session,
                         ID3D12CommandQueue* command_queue, uint32_t node_mask,
                         ID3D12Resource* side_by_side, bool flip_y_position,
                         DXGI_FORMAT format, ID3D12Resource* target);

/**
 * \~Japanese
 * @brief OpenGLのレンダリングテクスチャとシェーダの生成を行う。
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] side_by_side 両目テクスチャのポインタを渡す。
 * @param[in] flip_y_position VertexShaderのY軸を反転させるかを渡す。
 * @param[out] target Render Targetのポインタを渡す。
 * @return SonyOzResult
 * @details
 * オフスクリーン レンダリングとシェーダー用のテクスチャを生成する。
 * @attention
 * \~English
 * @brief Generate render textures and shaders for D3D11. (single pass)
 * @param[in] session Pass the session handle.
 * @param[in] side_by_side Pass a texture id of the SideBySide texture.
 * @param[in] flip_y_position Pass whether to invert the Y axis of the VertexShader.
 * @param[out] target Pass a texture id of the Render Target.
 * @return SonyOzResult
 * @details
 * Generate textures for off-screen rendering and shaders.
 * @attention
 */
SonyOzResult SubmitOpengl(SonyOzSessionHandle session,
                          unsigned int side_by_side, bool flip_y_position,
                          unsigned int target);

/**
 * \~Japanese
 * @brief D3D11のレンダリングテクスチャとシェーダの生成を行う。（マルチパス）
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] device
 * @param[in] left 左目テクスチャのポインタを渡す。
 * @param[in] right 右目テクスチャのポインタを渡す。
 * @param[in] flip_y_position VertexShaderのY軸を反転させるかを渡す。
 * @param[out] target Render Targetのポインタを渡す。
 * @return SonyOzResult
 * @details
 * オフスクリーン レンダリングとシェーダー用のテクスチャを生成する。
 * @attention
 * \~English
 * @brief Generate D3D11 render textures and shaders for off-screen rendering and shaders. (multi-pass)
 * @param[in] session Pass the session handle.
 * @param[in] device
 * @param[in] left Pass a pointer to the left eye texture.
 * @param[in] right Pass a pointer to the right eye texture.
 * @param[in] flip_y_position Pass whether to invert the Y axis of the VertexShader.
 * @param[out] target Pass a pointer to the Render Target.
 * @return SonyOzResult
 * @details
 * Generate textures for off-screen rendering and shaders.
 * @attention
 */
SonyOzResult SubmitMultiPassD3d11(SonyOzSessionHandle session,
                                  ID3D11Device* device, ID3D11Texture2D* left,
                                  ID3D11Texture2D* right, bool flip_y_position,
                                  ID3D11Texture2D* target);

/**
 * \~Japanese
 * @brief D3D12のレンダリングテクスチャとシェーダの生成を行う。（マルチパス）
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] command_queue
 * @param[in] node_mask
 * @param[in] left 左目テクスチャのポインタを渡す。
 * @param[in] right 右目テクスチャのポインタを渡す。
 * @param[in] flip_y_position VertexShaderのY軸を反転させるかを渡す。
 * @param[in] format
 * @param[out] target Render Targetのポインタを渡す。
 * @return SonyOzResult
 * @details
 * オフスクリーン レンダリングとシェーダー用のテクスチャを生成する。
 * @attention
 * \~English
 * @brief Generate D3D12 render textures and shaders for off-screen rendering and shaders. (multi-pass)
 * @param[in] session Pass the session handle.
 * @param[in] command_queue
 * @param[in] node_mask
 * @param[in] left Pass a pointer to the left eye texture.
 * @param[in] right Pass a pointer to the right eye texture.
 * @param[in] flip_y_position Pass whether to invert the Y axis of the VertexShader.
 * @param[in] format
 * @param[out] target Pass a pointer to the Render Target.
 * @return SonyOzResult
 * @details
 * Generate textures for off-screen rendering and shaders.
 * @attention
 */
SonyOzResult SubmitMultiPassD3d12(SonyOzSessionHandle session,
                                  ID3D12CommandQueue* command_queue,
                                  uint32_t node_mask, ID3D12Resource* left,
                                  ID3D12Resource* right, bool flip_y_position,
                                  DXGI_FORMAT format, ID3D12Resource* target);

/**
 * \~Japanese
 * @brief OpenGLのレンダリングテクスチャとシェーダの生成を行う。（マルチパス）
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] left 左目テクスチャのポインタを渡す。
 * @param[in] right 右目テクスチャのポインタを渡す。
 * @param[in] flip_y_position VertexShaderのY軸を反転させるかを渡す。
 * @param[out] target Render Targetのポインタを渡す。
 * @return SonyOzResult
 * @details
 * オフスクリーン レンダリングとシェーダー用のテクスチャを生成する。
 * @attention
 * \~English
 * @brief Generate render textures and shaders for D3D11. (single pass)
 * @param[in] session Pass the session handle.
 * @param[in] left Pass a pointer to the left eye texture.
 * @param[in] right Pass a pointer to the right eye texture.
 * @param[in] flip_y_position Pass whether to invert the Y axis of the VertexShader.
 * @param[out] target Pass a texture id of the Render Target.
 * @return SonyOzResult
 * @details
 * Generate textures for off-screen rendering and shaders.
 * @attention
 */
SonyOzResult SubmitMultiPassOpengl(SonyOzSessionHandle session,
                                   unsigned int left, unsigned int right,
                                   bool flip_y_position, unsigned int target);

/******************************************************************************
 * Data set/get process
 */

 /**
  * \~Japanese
  * @brief ステレオ描画の有効/無効を切り替える。
  * @param[in] session セッションのハンドルを渡す。
  * @param[in] enable true: 有効にする, false: 無効にする。
  * @return SonyOzResult
  * @details
  * ステレオ描画の有効/無効を切り替える。
  * @attention
  * \~English
  * @brief Enables/disables stereo rendering.
  * @param[in] session Pass the session handle.
  * @param[in] enable true: enable, false: disable.
  * @return SonyOzResult
  * @details
  * Enables/disables stereo rendering.
  * @attention
  */
SonyOzResult EnableStereo(SonyOzSessionHandle session, bool enable);


/**
 * \~Japanese
 * @brief ターゲットモニターの情報を取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[out] rect ウィンドウ情報を返す。
 * @return SonyOzResult
 * @details
 * @attention
 * \~English
 * @brief Obtain information on the target monitor.
 * @param[in] session Pass the session handle.
 * @param[out] rect Window information is returned.
 * @return SonyOzResult
 * @details
 * @attention
 */
SonyOzResult GetTargetMonitorRectangle(SonyOzSessionHandle session,
                                       SonyOzRect* rect);

/**
 * \~Japanese
 * @brief ディスプレイの情報を取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[out] display_spec ディスプレイの情報が返される。
 * @return SonyOzResult
 * @details
 * ディスプレイの寸法や解像度、傾きといった情報を取得する。
 * @attention
 * \~English
 * @brief Obtain information about the display.
 * @param[in] session Pass the session handle.
 * @param[out] display_spec Display information is returned.
 * @return SonyOzResult
 * @details
 * Obtain information such as display dimensions, resolution, and tilt.
 * @attention
 */
SonyOzResult GetDisplaySpec(SonyOzSessionHandle session,
                            SonyOzDisplaySpec* display_spec);

/******************************************************************************
 * System error process
 */

/**
 * \~Japanese
 * @brief XR Runtimeで発生したエラーを取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[out] error 最初に発生したエラーが返される。
 * @return SonyOzResult
 * @details
 * XR Runtimeで最初に発生したエラーを取得する。
 * @attention
 * \~English
 * @brief Obtain an error occurred in XR Runtime.
 * @param[in] session Pass the session handle.
 * @param[out] error The first error is returned.
 * @return SonyOzResult
 * @details
 * Obtain an error occurred in XR Runtime.
 * @attention
 */
SonyOzResult GetXrSystemError(SonyOzSessionHandle session,
                              SonyOzXrSystemError* error);

/**
 * \~Japanese
 * @brief XR Runtimeで発生したエラーの数を取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[out] num エラーの数が返される。
 * @return SonyOzResult
 * @details
 * XR Runtimeで発生したエラーの数を取得する。
 * @attention
 * \~English
 * @brief Obtain the number of errors occured in XR Runtime.
 * @param[in] session Pass the session handle.
 * @param[out] num The number of errors is returned.
 * @return SonyOzResult
 * @details
 * Obtain the number of errors occured in XR Runtime.
 * @attention
 */
SonyOzResult GetXrSystemErrorNum(SonyOzSessionHandle session, uint16_t* num);

/**
 * \~Japanese
 * @brief XR Runtimeで発生したエラーを取得する。
 * @param[in] session セッションのハンドルを渡す。
 * @param[in] num リストのサイズを渡す。
 * @param[out] errors 発生したエラーがリストへ格納される。
 * @return SonyOzResult
 * @details
 * XR Runtimeで発生したエラーを渡されたリストへ格納する。
 * @attention
 * numにはsony_ozGetXrSystemErrorNumで取得したサイズ以下を指定すること。
 * \~English
 * @brief Obtain errors occurred in XR Runtime.
 * @param[in] session Pass the session handle.
 * @param[in] num Pass the list size.
 * @param[out] errors The errors are stored in the list.
 * @return SonyOzResult
 * @details
 * Obtain errors occurred in XR Runtime.
 * @attention
 * num must be less than or equal to the size returned by GetXrSystemErrorNum.
 */
SonyOzResult GetXrSystemErrorList(SonyOzSessionHandle session, uint16_t num,
                                  SonyOzXrSystemError* errors);

SonyOzResult SetColorSpace(SonyOzSessionHandle session, int input_gamma_count,
    int output_gamma_count, float gamma);
}  // namespace sony::oz::xr_runtime

#endif  // XR_API_WRAPPER_H_
